/* 
Copyright (C) 1993 Free Software Foundation

This file is part of the GNU IO Library.  This library is free
software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option)
any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

As a special exception, if you link this library with files
compiled with a GNU compiler to produce an executable, this does not cause
the resulting executable to be covered by the GNU General Public License.
This exception does not however invalidate any other reasons why
the executable file might be covered by the GNU General Public License. */

/* Generic or default I/O operations. */

#include "libioP.h"

void _IO_un_link(fp)
     _IO_FILE *fp;
{
  if (fp->_flags & _IO_LINKED) {
    _IO_FILE **f;
    for (f = &_IO_list_all; *f != NULL; f = &(*f)->_chain) {
      if (*f == fp) {
	*f = fp->_chain;
	break;
      }
    }
    fp->_flags &= ~_IO_LINKED;
  }
}

void _IO_link_in(fp)
     _IO_FILE *fp;
{
    if ((fp->_flags & _IO_LINKED) == 0) {
	fp->_flags |= _IO_LINKED;
	fp->_chain = _IO_list_all;
	_IO_list_all = fp;
    }
}

/* Return minimum _pos markers
   Assumes the current get area is the main get area. */

_IO_size_t
_IO_least_marker(fp)
     register _IO_FILE *fp;
{
  _IO_ssize_t least_so_far = fp->_egptr - fp->_eback;
  register struct _IO_marker *mark;
  for (mark = fp->_markers; mark != NULL; mark = mark->_next)
    if (mark->_pos < least_so_far)
      least_so_far = mark->_pos;
  return least_so_far;
}

/* Switch current get area from backup buffer to (start of) main get area. */

void
_IO_switch_to_main_get_area(fp)
     _IO_FILE *fp;
{
  char *tmp;
  fp->_flags &= ~_IO_IN_BACKUP;
  /* Swap _egptr and _IO_save_end. */
  tmp = fp->_egptr; fp->_egptr= fp->_IO_save_end; fp->_IO_save_end= tmp;
  /* Swap _eback and _IO_save_base. */
  tmp = fp->_eback; fp->_eback = fp->_IO_save_base; fp->_IO_save_base = tmp;
  fp->_gptr = fp->_eback;
}

/* Switch current get area from main get area to (end of) backup area. */

void
_IO_switch_to_backup_area(fp)
     register _IO_FILE *fp;
{
  char *tmp;
  fp->_flags |= _IO_IN_BACKUP;
  /* Swap _egptr and _IO_save_end. */
  tmp = fp->_egptr; fp->_egptr = fp->_IO_save_end; fp->_IO_save_end = tmp;
  /* Swap _gbase and _IO_save_base. */
  tmp = fp->_eback; fp->_eback = fp->_IO_save_base; fp->_IO_save_base = tmp;
  fp->_gptr = fp->_egptr;
}

int
_IO_switch_to_get_mode(fp)
     register _IO_FILE *fp;
{
  if (fp->_pptr > fp->_pbase)
    if (fp->_jumps->__overflow(fp, EOF) == EOF)
      return EOF;
  if (_IO_in_backup(fp))
    fp->_eback = fp->_aux_limit;
  else
    {
      fp->_eback = fp->_base;
      if (fp->_pptr > fp->_egptr)
	fp->_egptr = fp->_pptr;
    }
  fp->_gptr = fp->_pptr;

  fp->_pbase = fp->_pptr = fp->_epptr = fp->_gptr;

  fp->_flags &= ~_IO_CURRENTLY_PUTTING;
  return 0;
}

void
_IO_free_backup_area(fp)
     register _IO_FILE *fp;
{
  if (_IO_in_backup (fp))
    _IO_switch_to_main_get_area(fp);  /* Just in case. */
  free (fp->_IO_save_base);
  fp->_IO_save_base = NULL;
  fp->_IO_save_end = NULL;
  fp->_aux_limit = NULL;
}

#if 0
int
_IO_switch_to_put_mode(fp)
     register _IO_FILE *fp;
{
  fp->_pbase = fp->_gptr;
  fp->_pptr = fp->_gptr;
  /* Following is wrong if line- or un-buffered? */
  fp->_epptr = fp->_flags & _IO_IN_BACKUP ? fp->_egptr : fp->_ebuf;

  fp->_gptr = fp->_egptr;
  fp->_eback = fp->_egptr;

  fp->_flags |= _IO_CURRENTLY_PUTTING;
  return 0;
}
#endif

int  __overflow(f, ch)
     _IO_FILE *f;
     int ch;
{
  return f->_jumps->__overflow(f, ch);
}

int
__underflow(fp)
     _IO_FILE *fp;
{
  if (_IO_in_put_mode(fp))
    if (_IO_switch_to_get_mode(fp) == EOF) return EOF;
  if (fp->_gptr < fp->_egptr)
    return *(unsigned char*)fp->_gptr;
  if (_IO_in_backup(fp))
    {
      _IO_switch_to_main_get_area(fp);
      if (fp->_gptr < fp->_egptr)
	return *fp->_gptr;
    }
  if (_IO_have_markers(fp))
    {
      /* Append [_gbase.._egptr] to backup area. */
      int least_mark = _IO_least_marker(fp);
      /* needed_size is how much space we need in the backup area. */
      int needed_size = (fp->_egptr - fp->_eback) - least_mark;
      int current_Bsize = fp->_IO_save_end - fp->_IO_save_base;
      int avail; /* Extra space available for future expansion. */
      int delta;
      struct _IO_marker *mark;
      if (needed_size > current_Bsize)
	{
	  char *new_buffer;
	  avail = 100;
	  new_buffer = (char*)malloc(avail+needed_size);
	  if (new_buffer == NULL)
	    return EOF; /* FIXME */
	  if (least_mark < 0)
	    {
	      memcpy(new_buffer + avail,
		     fp->_IO_save_end + least_mark,
		     -least_mark);
	      memcpy(new_buffer +avail - least_mark,
		     fp->_eback,
		     fp->_egptr - fp->_eback);
	    }
	  else
	    memcpy(new_buffer + avail,
		   fp->_eback + least_mark,
		   needed_size);
	  free (fp->_IO_save_base);
	  fp->_IO_save_base = new_buffer;
	  fp->_IO_save_end = new_buffer + avail + needed_size;
	}
      else
	{
	  avail = current_Bsize - needed_size;
	  if (least_mark < 0)
	    {
	      memmove(fp->_IO_save_base + avail,
		      fp->_IO_save_end + least_mark,
		      -least_mark);
	      memcpy(fp->_IO_save_base + avail - least_mark,
		     fp->_eback,
		     fp->_egptr - fp->_eback);
	    }
	  else if (needed_size > 0)
	    memcpy(fp->_IO_save_base + avail,
		   fp->_eback + least_mark,
		   needed_size);
	}
      /* FIXME: Dubious arithmetic if pointers are NULL */
      fp->_aux_limit = fp->_IO_save_base + avail;
      /* Adjust all the streammarkers. */
      delta = fp->_egptr - fp->_eback;
      for (mark = fp->_markers; mark != NULL; mark = mark->_next)
	mark->_pos -= delta;
    }
  else if (_IO_have_backup(fp))
    _IO_free_backup_area(fp);
  return fp->_jumps->__underflow(fp);
}

void
_IO_setb(f, b, eb, a)
     _IO_FILE *f;
     char *b;
     char *eb;
     int a;
{
  if (f->_base && !(f->_flags & _IO_USER_BUF))
    FREE_BUF(f->_base);
  f->_base = b;
  f->_ebuf = eb;
  if (a)
    f->_flags &= ~_IO_USER_BUF;
  else
    f->_flags |= _IO_USER_BUF;
}

void
_IO_doallocbuf(fp)
     register _IO_FILE *fp;
{
  if (fp->_base)
    return;
  if (!(fp->_flags & _IO_UNBUFFERED))
    if (fp->_jumps->__doallocate(fp) != EOF)
      return;
  _IO_setb(fp, fp->_shortbuf, fp->_shortbuf+1, 0);
}

_IO_size_t
_IO_default_xsputn(f, data, n)
     register _IO_FILE *f;
     const void *data;
     _IO_size_t n;
{
  register const char *s = data;
  register _IO_size_t more = n;
  if (more <= 0)
    return 0;
  for (;;)
    {
      _IO_size_t count = f->_epptr - f->_pptr; /* Space available. */
      if (count > 0)
	{
	  if (count > more)
	    count = more;
	  if (count > 20)
	    {
	      memcpy(f->_pptr, s, count);
	      s += count;
	      f->_pptr += count;
            }
	  else if (count <= 0)
	    count = 0;
	  else
	    {
	      register char *p = f->_pptr;
	      register _IO_ssize_t i;
	      for (i = count; --i >= 0; ) *p++ = *s++;
	      f->_pptr = p;
            }
	  more -= count;
        }
      if (more == 0 || __overflow(f, (unsigned char)*s++) == EOF)
	break;
      more--;
    }
  return n - more;
}

_IO_size_t
_IO_sgetn(fp, data, n)
     _IO_FILE *fp;
     void *data;
     _IO_size_t n;
{
  /* FIXME handle putback buffer here! */
  return fp->_jumps->__xsgetn(fp, data, n);
}

_IO_size_t
_IO_default_xsgetn(fp, data, n)
     _IO_FILE *fp;
     void *data;
     _IO_size_t n;
{
  register _IO_size_t more = n;
  register char *s = data;
  for (;;)
    {
      _IO_size_t count = fp->_egptr - fp->_gptr; /* Data available. */
      if (count > 0)
	{
	  if (count > more)
	    count = more;
	  if (count > 20)
	    {
	      memcpy(s, fp->_gptr, count);
	      s += count;
	      fp->_gptr += count;
	    }
	  else if (count <= 0)
	    count = 0;
	  else
	    {
	      register char *p = fp->_gptr;
	      register int i = (int)count;
	      while (--i >= 0) *s++ = *p++;
	      fp->_gptr = p;
            }
            more -= count;
        }
      if (more == 0 || __underflow(fp) == EOF)
	break;
    }
  return n - more;
}

int
_IO_sync(fp)
     register _IO_FILE *fp;
{
  if (fp->_gptr == fp->_egptr && fp->_pptr == fp->_pbase)
    return 0;
  return EOF;
}

int
_IO_default_setbuf(fp, p, len)
     register _IO_FILE *fp;
     char* p;
     _IO_ssize_t len;
{
    if (fp->_jumps->__sync(fp) == EOF)
	return EOF;
    if (p == NULL || len == 0)
      {
	fp->_flags |= _IO_UNBUFFERED;
	_IO_setb(fp, fp->_shortbuf, fp->_shortbuf+1, 0);
      }
    else
      {
	fp->_flags &= ~_IO_UNBUFFERED;
	_IO_setb(fp, p, p+len, 0);
      }
    fp->_pbase = fp->_pptr = fp->_epptr = 0;
    fp->_eback = fp->_gptr = fp->_egptr = 0;
    return 0;
}

_IO_pos_t
_IO_default_seekpos(fp, pos, mode)
     _IO_FILE *fp;
     _IO_pos_t pos;
     _IO_seekflags mode;
{
  return fp->_jumps->__seekoff(fp, _IO_pos_as_off(pos), mode);
}

int
_IO_default_doallocate(fp)
     _IO_FILE *fp;
{
  char *buf = ALLOC_BUF(_IO_BUFSIZ);
  if (buf == NULL)
    return EOF;
  _IO_setb(fp, buf, buf+_IO_BUFSIZ, 1);
  return 1;
}

void
_IO_init(fp, flags)
     register _IO_FILE *fp;
     int flags;
{
  fp->_flags = _IO_MAGIC|flags;
  fp->_base = NULL;
  fp->_ebuf = NULL;
  fp->_eback = NULL;
  fp->_gptr = NULL;
  fp->_egptr = NULL;
  fp->_pbase = NULL;
  fp->_pptr = NULL;
  fp->_epptr = NULL;
  fp->_chain = NULL; /* Not necessary. */

  fp->_IO_save_base = NULL;
  fp->_aux_limit = NULL;
  fp->_IO_save_end = NULL;
  fp->_markers = NULL;
  fp->_cur_column = 0;
}

void _IO_default_finish (fp)
     _IO_FILE *fp;
{
  struct _IO_marker *mark;
  if (fp->_base && !(fp->_flags & _IO_USER_BUF))
    FREE_BUF(fp->_base);

  for (mark = fp->_markers; mark != NULL; mark = mark->_next)
    mark->_sbuf = NULL;

  free (fp->_IO_save_base);

  _IO_un_link(fp);
}

_IO_pos_t
_IO_default_seekoff(fp, offset, mode)
     register _IO_FILE *fp;
     _IO_off_t offset;
     _IO_seekflags mode;
{
    return _IO_pos_BAD;
}

int
_IO_sputbackc(fp, c) /* ungetc(c, fp) */
     register _IO_FILE *fp;
     int c;
{
  if (fp->_gptr > fp->_eback
      && (unsigned char)fp->_gptr[-1] == (unsigned char)c)
    {
      fp->_gptr--;
      return (unsigned char)c;
    }
  return fp->_jumps->__pbackfail(fp, c);
}

int
_IO_sungetc(fp)
     register _IO_FILE *fp;
{
  if (fp->_gptr > fp->_eback)
    {
      fp->_gptr--;
      return (unsigned char)*fp->_gptr;
    }
  else
    return fp->_jumps->__pbackfail(fp, EOF);
}

#if 0 /* Work in progress */
void
_IO_set_column(fp, c)
     register _IO_FILE *fp;
     int c;
{
  if (c == -1)
    fp->_collumn = -1;
  else
    fp->_collumn = c - (fp->_pptr - fp->_pbase);
}
#else
int
_IO_set_column(fp, i)
     register _IO_FILE *fp;
     int i;
{
  fp->_cur_column = i+1;
  return 0;
}
#endif


unsigned
_IO_adjust_column(start, line, count)
     unsigned start;
     const char *line;
     int count;
{
  register const char *ptr = line + count;
  while (ptr > line)
    if (*--ptr == '\n')
      return line + count - ptr - 1;
  return start + count;
}

int
_IO_get_column(fp)
     register _IO_FILE *fp;
{
  if (fp->_cur_column) 
    return _IO_adjust_column(fp->_cur_column - 1,
			      fp->_pbase,
			      fp->_pptr - fp->_pbase);
  return -1;
}

int
_IO_flush_all()
{
  int result = 0;
  _IO_FILE *fp;
  for (fp = _IO_list_all; fp != NULL; fp = fp->_chain)
    if (fp->_jumps->__overflow(fp, EOF) == EOF)
      result = EOF;
  return result;
}

void
_IO_flush_all_linebuffered()
{
  _IO_FILE *fp;
  for (fp = _IO_list_all; fp != NULL; fp = fp->_chain)
    if (fp->_flags & _IO_LINE_BUF)
      fp->_jumps->__overflow(fp, EOF);
}

void _IO_init_marker(marker, fp)
     struct _IO_marker *marker;
     _IO_FILE *fp;
{
  marker->_sbuf = fp;
  if (_IO_in_put_mode(fp))
    _IO_switch_to_get_mode(fp);
  if (_IO_in_backup(fp))
    marker->_pos = fp->_IO_read_ptr - fp->_IO_read_end;
  else
    marker->_pos = fp->_IO_read_ptr - fp->_IO_read_base;
  
  /* Should perhaps sort the chain? */
  marker->_next = fp->_markers;
  fp->_markers = marker;
}

void
_IO_remove_marker (marker)
     register struct _IO_marker * marker;
{
  /* Unlink from sb's chain. */
  register struct _IO_marker **ptr = &marker->_sbuf->_markers;
  for (; ; ptr = &(*ptr)->_next)
    {
      if (*ptr == NULL)
	break;
      else if (*ptr == marker)
	{
	  *ptr = marker->_next;
	  return;
	}
    }
#if 0
    if _sbuf has a backup area that is no longer needed, should we delete
    it now, or wait until the next underflow?
#endif
}

#define BAD_DELTA EOF

int
_IO_marker_difference(mark1, mark2)
     struct _IO_marker *mark1;
     struct _IO_marker *mark2;
{
  return mark1->_pos - mark2->_pos;
}

/* Return difference between MARK and current posistion of MARK's stream. */
int
_IO_marker_delta(mark)
     struct _IO_marker *mark;
{
  int cur_pos;
  if (mark->_sbuf == NULL)
    return BAD_DELTA;
  if (_IO_in_backup(mark->_sbuf))
    cur_pos = mark->_sbuf->_IO_read_ptr - mark->_sbuf->_IO_read_end;
  else
    cur_pos = mark->_sbuf->_IO_read_ptr - mark->_sbuf->_IO_read_base;
  return mark->_pos - cur_pos;
}

int _IO_seekmark(fp, mark, delta)
     _IO_FILE *fp;
     struct _IO_marker *mark;
     int delta;
{
  if (mark->_sbuf != fp)
    return EOF;
 if (mark->_pos >= 0)
    {
      if (_IO_in_backup(fp))
	_IO_switch_to_main_get_area(fp);
      fp->_gptr = fp->_eback + mark->_pos;
    }
  else
    {
      if (!_IO_in_backup(fp))
	_IO_switch_to_backup_area(fp);
      fp->_gptr = fp->_egptr + mark->_pos;
    }
}

void _IO_unsave_markers(fp)
     register _IO_FILE *fp;
{
  register struct _IO_marker *mark = fp->_markers;
  if (mark)
    {
#ifdef TODO
      streampos offset = seekoff(0, ios::cur, ios::in);
      if (offset != EOF)
	{
	  offset += eGptr() - Gbase();
	  for ( ; mark != NULL; mark = mark->_next)
	    mark->set_streampos(mark->_pos + offset);
	}
    else
      {
	for ( ; mark != NULL; mark = mark->_next)
	  mark->set_streampos(EOF);
      }
#endif
      fp->_markers = 0;
    }

  _IO_free_backup_area(fp);
}

int
_IO_nobackup_pbackfail(fp, c)
     register _IO_FILE *fp;
     int c;
{
  if (fp->_gptr > fp->_eback)
	fp->_gptr--;
  if (c != EOF && *fp->_gptr != c)
      *fp->_gptr = c;
  return (unsigned char)c;
}

int
_IO_default_pbackfail(fp, c)
     register _IO_FILE *fp;
     int c;
{
  if (fp->_gptr <= fp->_eback)
      {
	/* Need to handle a filebuf in write mode (switch to read mode). FIXME!*/
	if (_IO_have_backup(fp) && !_IO_in_backup(fp))
	  _IO_switch_to_backup_area(fp);
	
	if (!_IO_have_backup(fp))
	  {
	    /* No backup buffer: allocate one. */
	    /* Use nshort buffer, if unused? (probably not)  FIXME */
	    int backup_size = 128;
	    char *bbuf = (char*)malloc(backup_size);
	    if (bbuf == NULL)
	      return EOF;
	    fp->_IO_save_base = bbuf;
	    fp->_IO_save_end = fp->_IO_save_base + backup_size;
	    fp->_aux_limit = fp->_IO_save_end;
	    _IO_switch_to_backup_area(fp);
	  }
	else if (fp->_gptr <= fp->_eback)
	  {
	    /* Increase size of existing backup buffer. */
	    _IO_size_t new_size;
	    _IO_size_t old_size = fp->_egptr - fp->_eback;
	    char *new_buf;
	    new_size = 2 * old_size;
	    new_buf = (char*)malloc(new_size);
	    if (new_buf == NULL)
	      return EOF;
	    memcpy(new_buf+(new_size-old_size), fp->_eback, old_size);
	    free (fp->_eback);
	    _IO_setg(fp,
		     new_buf, new_buf+(new_size-old_size), new_buf+new_size);
	    fp->_aux_limit = fp->_gptr;
	  }
      }
  fp->_gptr--;
  if (c != EOF && *fp->_gptr != c)
    *fp->_gptr = c;
  return (unsigned char)*fp->_gptr;
}

_IO_pos_t
_IO_default_seek(fp, offset, dir)
     _IO_FILE *fp;
     _IO_off_t offset;
     int dir;
{
  return _IO_pos_BAD;
}

int
_IO_default_stat(fp, st)
     _IO_FILE *fp;
     void* st;
{
  return EOF;
}

_IO_ssize_t
_IO_default_read(fp, data, n)
     register _IO_FILE* fp;
     void* data;
     _IO_ssize_t n;
{
  return -1;
}

_IO_ssize_t
_IO_default_write(fp, data, n)
     register _IO_FILE* fp;
     const void* data;
     _IO_ssize_t n;
{
  return 0;
}


#ifdef TODO
#if defined(linux)
#define IO_CLEANUP ;
#endif

#ifdef IO_CLEANUP
  IO_CLEANUP
#else
struct __io_defs {
    __io_defs() { }
    ~__io_defs() { _IO_flush_all(); }
};   
__io_defs io_defs__;
#endif

#endif /* TODO */
