// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/


#ifndef _String_h
#pragma once
#define _String_h 1

#include <stream.h>

struct StrRep                     // internal String representations
{
  unsigned short    len;         // string length 
  unsigned short    sz;          // allocated space
  char              s[1];        // the string starts here 
                                 // (at least 1 char for trailing null)
                                 // allocated & expanded via non-public fcts
};

StrRep*     Salloc(StrRep*, const char*, int, int);
StrRep*     Scopy(StrRep*, StrRep*);
StrRep*     Sresize(StrRep*, int);
StrRep*     Scat(StrRep*, const char*, int, const char*, int);
StrRep*     Sprepend(StrRep*, const char*, int);
StrRep*     Sreverse(StrRep*, StrRep*);
StrRep*     Supcase(StrRep*, StrRep*);
StrRep*     Sdowncase(StrRep*, StrRep*);
StrRep*     Scapitalize(StrRep*, StrRep*);

class String;
class SubString;
class StrTmp;


struct re_pattern_buffer;       // defined elsewhere
struct re_registers;

class Regex
{
  friend class       String;
  friend class       SubString;

                     Regex(const& Regex) {}  // no X(X&)
  void               operator = (const Regex&) {} // no assignment

protected:
  re_pattern_buffer* buf;
  re_registers*      reg;

  void               initialize(const char* t, int tlen, int fast, 
                                int bufsize, const char* transtable);

public:
                     Regex(const char* t, 
                           int fast = 0, 
                           int bufsize = 40, 
                           const char* transtable = 0);

                     Regex(const String& x, 
                           int fast = 0, 
                           int bufsize = 40, 
                           const char* transtable = 0);

                    ~Regex();

  int                match(const char* s, int len, int pos = 0) const;
  int                search(const char* s, int len, 
                            int& matchlen, int startpos = 0) const;
  int                match_info(int& start, int& length, int nth = 0) const;

  int                OK() const;  // representation invariant
};



class SubString
{
  friend class      String;
  friend class      StrTmp;
protected:
  String&           S;
  unsigned short    pos;
  unsigned short    len;

  void              assign(StrRep*, const char*, int = -1);
                    SubString(String& x, int p, int l);
                    SubString(const SubString& x);

public:
                   ~SubString();

  void              operator =  (const String&     y);
  void              operator =  (const SubString&  y);
  void              operator =  (const char* t);
  void              operator =  (char        c);
  
  StrTmp            operator +  (const String&     y) const;
  StrTmp            operator +  (const SubString&  y) const;
  StrTmp            operator +  (const char* t) const;
  StrTmp            operator +  (char        c) const;
  StrTmp            operator +  (StrTmp& y) const;

  int               contains(char        c) const;
  int               contains(const String&     y) const;
  int               contains(const SubString&  y) const;
  int               contains(const char* t) const;
  int               contains(const Regex&       r) const;

  int               matches(const Regex&  r) const;

// misc

  int               length() const;
  int               empty() const;

  friend ostream&   operator<<(ostream& s, const SubString& x);

  const char*       chars() const;

  int               OK() const; 

};


class String
{
  friend class      SubString;
  friend class      Regex;
  friend class      StrTmp;

protected:
  StrRep*           rep;

  int               search(int, int, const char*, int = -1) const;
  int               search(int, int, char) const;
  int               match(int, int, int, const char*, int = -1) const;
  int               _gsub(const char*, int, const char* ,int);
  int               _gsub(const Regex&, const char*, int);

  SubString         _substr(int, int);

public:

// constructors & assignment

                    String();
                    String(const String& x);
                    String(const SubString&  x);
                    String(const char* t);
                    String(const char* t, int len);
                    String(char c);

                    ~String();

  void              operator =  (const String&     y);
  void              operator =  (StrTmp&     y);
  void              operator =  (const char* y);
  void              operator =  (char        c);
  void              operator =  (const SubString&  y);

// concatenation

  StrTmp            operator +  (const String&     y) const;     
  StrTmp            operator +  (const char* t) const;
  StrTmp            operator +  (char        c) const;
  StrTmp            operator +  (const SubString&  y) const;     
  StrTmp            operator +  (StrTmp& y) const;

  void              operator += (const String&     y); 
  void              operator += (const SubString&  y);
  void              operator += (const char* t);
  void              operator += (char        c);

  void              prepend(const String&     y); 
  void              prepend(const SubString&  y);
  void              prepend(const char* t);
  void              prepend(char        c);

// searching & matching

  int               index(char        c, int startpos = 0) const;      
  int               index(const String&     y, int startpos = 0) const;      
  int               index(const SubString&  y, int startpos = 0) const;      
  int               index(const char* t, int startpos = 0) const;  
  int               index(const Regex&      r, int startpos = 0) const;       

  int               contains(char        c) const;
  int               contains(const String&     y) const;
  int               contains(const SubString&  y) const;
  int               contains(const char* t) const;
  int               contains(const Regex&      r) const;

  int               contains(char        c, int pos) const;
  int               contains(const String&     y, int pos) const;
  int               contains(const SubString&  y, int pos) const;
  int               contains(const char* t, int pos) const;
  int               contains(const Regex&      r, int pos) const;

  int               matches(char        c, int pos = 0) const;
  int               matches(const String&     y, int pos = 0) const;
  int               matches(const SubString&  y, int pos = 0) const;
  int               matches(const char* t, int pos = 0) const;
  int               matches(const Regex&      r, int pos = 0) const;

  int               freq(char        c) const; //  number of c's in string
  int               freq(const String&     y) const;
  int               freq(const SubString&  y) const;
  int               freq(const char* t) const;

// substring extraction

  SubString         at(int         pos, int len);
  SubString         at(const String&     x, int startpos = 0); 
  SubString         at(const SubString&  x, int startpos = 0); 
  SubString         at(const char* t, int startpos = 0);
  SubString         at(char        c, int startpos = 0);
  SubString         at(const Regex&      r, int startpos = 0); 

  SubString         before(int          pos);
  SubString         before(const String&      x, int startpos = 0);
  SubString         before(const SubString&   x, int startpos = 0);
  SubString         before(const char*  t, int startpos = 0);
  SubString         before(char         c, int startpos = 0);
  SubString         before(const Regex&       r, int startpos = 0);

  SubString         through(int          pos);
  SubString         through(const String&      x, int startpos = 0);
  SubString         through(const SubString&   x, int startpos = 0);
  SubString         through(const char*  t, int startpos = 0);
  SubString         through(char         c, int startpos = 0);
  SubString         through(const Regex&       r, int startpos = 0);

  SubString         from(int          pos);
  SubString         from(const String&      x, int startpos = 0);
  SubString         from(const SubString&   x, int startpos = 0);
  SubString         from(const char*  t, int startpos = 0);
  SubString         from(char         c, int startpos = 0);
  SubString         from(const Regex&       r, int startpos = 0);

  SubString         after(int         pos);
  SubString         after(const String&     x, int startpos = 0);
  SubString         after(const SubString&  x, int startpos = 0);
  SubString         after(const char* t, int startpos = 0);
  SubString         after(char        c, int startpos = 0);
  SubString         after(const Regex&      r, int startpos = 0);


// modification

  void              del(int         pos, int len);
  void              del(const String&     y, int startpos = 0);
  void              del(const SubString&  y, int startpos = 0);
  void              del(const char* t, int startpos = 0);
  void              del(char        c, int startpos = 0);
  void              del(const Regex&      r, int startpos = 0);

  int               gsub(const String&     pat, const String&     repl);
  int               gsub(const SubString&  pat, const String&     repl);
  int               gsub(const char* pat, const String&     repl);
  int               gsub(const char* pat, const char* repl);
  int               gsub(const Regex&      pat, const String&     repl);

// friends & utilities

  friend int        split(const String& x, String res[], int maxn, 
                          const String& sep);
  friend int        split(const String& x, String res[], int maxn, 
                          const Regex&  sep);

  friend StrTmp     reverse(const String& x);
  friend StrTmp     upcase(const String& x);
  friend StrTmp     downcase(const String& x);
  friend StrTmp     capitalize(const String& x);

// in-place versions of above

  void              reverse();
  void              upcase();
  void              downcase();
  void              capitalize();

// conversion

  char&             operator [] (int i);
  char              elem(int i) const;
  char              firstchar() const;
  char              lastchar() const;

                    operator const char*() const;
  const char*       chars() const;


// IO

  friend ostream&   operator<<(ostream& s, const String& x);
  friend ostream&   operator<<(ostream& s, const SubString& x);
  friend istream&   operator>>(istream& s, String& x);

  friend int        readline(istream& s, String& x, 
                             char terminator = '\n',
                             int discard_terminator = 1);

// status

  int               length() const;
  int               empty() const;

  void              alloc(int newsize);
  int               allocation() const;

  volatile void     error(char* msg) const;

  int               OK() const;

};

class StrTmp : public String
{
public:
                    StrTmp(StrRep* p);
                    StrTmp(String& x);
                    StrTmp(StrTmp& x);
                    ~StrTmp();

  StrTmp            operator + (const String& y); 
  StrTmp            operator + (const SubString& y); 
  StrTmp            operator + (const char* y); 
  StrTmp            operator + (char y); 
  friend StrTmp     operator + (const char* x, StrTmp& y);
  friend StrTmp     operator + (char x, StrTmp& y);

  friend StrTmp     reverse(StrTmp& x);
  friend StrTmp     upcase(StrTmp& x);
  friend StrTmp     downcase(StrTmp& x);
  friend StrTmp     capitalize(StrTmp& x);

};

// other externs

int        compare(const String&    x, const String&     y);
int        compare(const String&    x, const SubString&  y);
int        compare(const String&    x, const char* y);
int        compare(const SubString& x, const String&     y);
int        compare(const SubString& x, const SubString&  y);
int        compare(const SubString& x, const char* y);
int        fcompare(const String&   x, const String&     y); // ignore case
StrTmp     common_prefix(const String& x, const String& y, int startpos = 0);
StrTmp     common_suffix(const String& x, const String& y, int startpos = -1);
StrTmp     replicate(char        c, int n);
StrTmp     replicate(const String&     y, int n);
StrTmp     join(String src[], int n, const String& sep);


// some built in regular expressions

extern const Regex RXwhite;          // = "[ \n\t\r\v\f]+"
extern const Regex RXint;            // = "-?[0-9]+"
extern const Regex RXdouble;         // = "-?\\(\\([0-9]+\\.[0-9]*\\)\\|
                                     //    \\([0-9]+\\)\\|\\(\\.[0-9]+\\)\\)
                                     //    \\([eE][---+]?[0-9]+\\)?"
extern const Regex RXalpha;          // = "[A-Za-z]+"
extern const Regex RXlowercase;      // = "[a-z]+"
extern const Regex RXuppercase;      // = "[A-Z]+"
extern const Regex RXalphanum;       // = "[0-9A-Za-z]+"
extern const Regex RXidentifier;     // = "[A-Za-z_][A-Za-z0-9_]*"


//#ifdef __OPTIMIZE__


extern StrRep  _nilStrRep;
extern String _nilString;

inline int String::length() const {  return rep->len; }

inline int String::empty() const { return rep->len == 0; }

inline  int SubString::length() const { return len; }

inline  int SubString::empty() const { return len == 0; }

inline const char* String::chars() const { return &(rep->s[0]); }

inline const char* SubString::chars() const { return &(S.rep->s[pos]); }

inline String::String() : rep(&_nilStrRep) {}

inline String::String(const String& x) : rep(Scopy(0, x.rep)) {}

inline String::String(const char* t) : rep(Salloc(0, t, -1, -1)) {}

inline StrTmp::StrTmp(StrRep* r) :  rep(r) {}

inline StrTmp::StrTmp(String& x) :rep(x.rep) { x.rep = &_nilStrRep; }

inline StrTmp::StrTmp(StrTmp& x) :rep(x.rep) { x.rep = &_nilStrRep; }

inline String::String(const char* t, int tlen)
  : rep(Salloc(0, t, tlen, tlen)) {}

inline String::String(const SubString& y)
 : rep(Salloc(0, y.chars(), y.length(), y.length())) {}

inline String::String(char c) : rep(Salloc(0, &c, 1, 1)) {}

inline String::~String() { if (rep != &_nilStrRep) delete rep; }

inline StrTmp::~StrTmp() {} 

inline SubString::SubString(const SubString& x)
  :S(x.S), pos(x.pos), len(x.len) {}

inline SubString::SubString(String& x, int first, int l)
  :S(x), pos(first), len(l) {}

inline SubString::~SubString() {}

inline void String::operator =  (const String& y)
{ 
  rep = Scopy(rep, y.rep);
}

inline void String::operator =  (StrTmp& y)
{ 
  if (rep != &_nilStrRep) delete rep; 
  rep = y.rep; y.rep = &_nilStrRep;
}

inline void String::operator=(const char* t)
{
  rep = Salloc(rep, t, -1, -1); 
}

inline void String::operator=(const SubString&  y)
{
  rep = Salloc(rep, y.chars(), y.length(), y.length());
}

inline void String::operator=(char c)
{
  rep = Salloc(rep, &c, 1, 1); 
}

inline void SubString::operator = (const char* ys)
{
  assign(0, ys);
}

inline void SubString::operator = (char ch)
{
  assign(0, &ch, 1);
}

inline void SubString::operator = (const String& y)
{
  assign(y.rep, y.chars(), y.length());
}

inline void SubString::operator = (const SubString& y)
{
  assign(y.S.rep, y.chars(), y.length());
}

inline void String::operator +=(const String& y)
{
  rep = Scat(rep, chars(), length(), y.chars(), y.length());
}

inline void String::operator +=(const SubString& y)
{
  rep = Scat(rep, chars(), length(), y.chars(),y.length());
}

inline void String::operator += (const char* y)
{
  rep = Scat(rep, chars(), length(), y, -1);
}

inline void String:: operator +=(char y)
{
  rep = Scat(rep, chars(), length(), &y, 1); 
}

inline StrTmp String::operator + (const String& y) const
{
  return(Scat(0, chars(), length(), y.chars(), y.length()));
}

inline StrTmp String::operator +(const SubString& y) const
{
  return(Scat(0, chars(), length(), y.chars(),y.length()));
}

inline StrTmp String::operator + (const char* y) const
{
  return(Scat(0, chars(), length(), y, -1));
}

inline StrTmp String::operator + (char y) const
{
  return(Scat(0, chars(), length(), &y, 1));
}

inline StrTmp SubString::operator + (const String& y) const
{
  return(Scat(0, chars(), len, y.chars(), y.length()));
}

inline StrTmp SubString::operator + (const SubString& y) const
{
  return(Scat(0, chars(), len, y.chars(), y.length()));
}

inline StrTmp SubString::operator + (const char* y) const
{
  return(Scat(0, chars(), len, y, -1));
}

inline StrTmp SubString::operator + (char y) const
{
  return(Scat(0, chars(), len, &y, 1));
}

inline StrTmp operator +(const char* t, const String& y)
{
  return(Scat(0, t, -1, y.chars(), y.length()));
}

inline StrTmp operator + (const char* t, const SubString& y)
{
  return(Scat(0, t, -1, y.chars(), y.length()));
}

inline StrTmp operator + (char c, const String& y)
{
  return(Scat(0, &c, 1, y.chars(), y.length()));
}

inline StrTmp operator + (char c, const SubString& y)
{
  return(Scat(0, &c, 1, y.chars(), y.length()));
}

inline StrTmp StrTmp::operator + (const String& y)
{
  rep = Scat(rep, chars(), length(), y.chars(), y.length()); return *this;
}

inline StrTmp StrTmp::operator + (const SubString& y)
{
  rep = Scat(rep, chars(), length(), y.chars(),y.length()); return *this;
}

inline StrTmp StrTmp::operator + (const char* y)
{
  rep = Scat(rep, chars(), length(), y, -1); return *this;
}

inline StrTmp StrTmp::operator + (char y)
{
  rep = Scat(rep, chars(), length(), &y, 1); return *this;
}

inline StrTmp String::operator + (StrTmp& y) const
{
  y.rep = Sprepend(y.rep, chars(), length()); return y;
}

inline StrTmp SubString::operator + (StrTmp& y) const
{
  y.rep = Sprepend(y.rep, chars(), len); return y;
}

inline StrTmp operator + (const char* x, StrTmp& y)
{
  y.rep = Sprepend(y.rep, x, -1); return y;
}

inline StrTmp operator + (char x, StrTmp& y)
{
  y.rep = Sprepend(y.rep, &x, 1); return y;
}

inline void String::prepend(const String& y)
{
  rep = Sprepend(rep, y.chars(), y.length());
}

inline void String::prepend(const char* y)
{
  rep = Sprepend(rep, y, -1); 
}

inline void String::prepend(char y)
{
  rep = Sprepend(rep, &y, 1); 
}

inline void String::prepend(const SubString& y)
{
  rep = Sprepend(rep, y.chars(), y.length());
}

inline StrTmp reverse(const String& x)
{
  return(Sreverse(x.rep, 0));
}

inline StrTmp reverse(StrTmp& x)
{
  x.rep = Sreverse(x.rep, x.rep); return x;
}

inline void String::reverse()
{
  rep = Sreverse(rep, rep);
}

inline StrTmp upcase(const String& x)
{
  return(Supcase(x.rep, 0));
}

inline StrTmp upcase(StrTmp& x)
{
  x.rep = Supcase(x.rep, x.rep); return x;
}

inline void String::upcase()
{
  rep = Supcase(rep, rep);
}

inline StrTmp downcase(const String& x)
{
  return(Sdowncase(x.rep, 0));
}

inline StrTmp downcase(StrTmp& x)
{
  x.rep = Sdowncase(x.rep, x.rep); return x;
}

inline void String::downcase()
{
  rep = Sdowncase(rep, rep);
}

inline StrTmp capitalize(const String& x)
{
  return(Scapitalize(x.rep, 0));
}

inline StrTmp capitalize(StrTmp& x)
{
  x.rep = Scapitalize(x.rep, x.rep); return x;
}

inline void String::capitalize()
{
  rep = Scapitalize(rep, rep);
}

inline void String::alloc(int newsize)
{
  rep = Sresize(rep, newsize);
}

inline int String::allocation() const
{
  return rep->sz;
}

inline char&  String::operator [] (int i) 
{ 
  if (((unsigned)i) >= length()) error("invalid index");
  return rep->s[i];
}

inline char  String::elem (int i) const
{ 
  if (((unsigned)i) >= length()) error("invalid index");
  return rep->s[i];
}

inline char  String::firstchar() const
{ 
  return elem(0);
}

inline char  String::lastchar() const
{ 
  return elem(length() - 1);
}

inline int String::index(char c, int startpos = 0) const
{
  return search(startpos, length(), c);
}

inline int String::index(const char* t, int startpos = 0) const
{   
  return search(startpos, length(), t);
}

inline int String::index(const String& y, int startpos = 0) const
{   
  return search(startpos, length(), y.chars(), y.length());
}

inline int String::index(const SubString& y, int startpos = 0) const
{   
  return search(startpos, length(), y.chars(), y.length());
}

inline int String::contains(char c) const
{
  return search(0, length(), c) >= 0;
}

inline int SubString::contains(char c) const
{
  return S.search(pos, pos+len, 0, c) >= 0;
}

inline int String::contains(const char* t) const
{   
  return search(0, length(), t) >= 0;
}

inline int String::contains(const String& y) const
{   
  return search(0, length(), y.chars(), y.length()) >= 0;
}

inline int String::contains(const SubString& y) const
{   
  return search(0, length(), y.chars(), y.length()) >= 0;
}

inline int SubString::contains(const char* t) const
{   
  return S.search(pos, pos+len, t) >= 0;
}

inline int SubString::contains(const String& y) const
{   
  return S.search(pos, pos+len, y.chars(), y.length()) >= 0;
}

inline int SubString::contains(const SubString&  y) const
{   
  return S.search(pos, pos+len, y.chars(), y.length()) >= 0;
}

inline int String::contains(char c, int p) const
{
  return match(p, length(), 0, &c, 1);
}

inline int String::matches(char c, int p = 0) const
{
  return match(p, length(), 1, &c, 1);
}

inline int String::contains(const char* t, int p) const
{
  return match(p, length(), 0, t);
}

inline int String::matches(const char* t, int p = 0) const
{
  return match(p, length(), 1, t);
}

inline int String::contains(const String& y, int p) const
{
  return match(p, length(), 0, y.chars(), y.length());
}

inline int String::matches(const String& y, int p = 0) const
{
  return match(p, length(), 1, y.chars(), y.length());
}

inline int String::contains(const SubString& y, int p) const
{
  return match(p, length(), 0, y.chars(), y.length());
}

inline int String::matches(const SubString& y, int p = 0) const
{
  return match(p, length(), 1, y.chars(), y.length());
}

inline int String::contains(const Regex& r) const
{
  int unused;  return r.search(chars(), length(), unused, 0) >= 0;
}

inline int SubString::contains(const Regex& r) const
{
  int unused;  return r.search(chars(), len, unused, 0) >= 0;
}

inline int String::contains(const Regex& r, int p) const
{
  return r.match(chars(), length(), p) >= 0;
}

inline int String::matches(const Regex& r, int p = 0) const
{
  int l = (p < 0)? -p : length() - p;
  return r.match(chars(), length(), p) == l;
}

inline int SubString::matches(const Regex& r) const
{
  return r.match(chars(), len, 0) == len;
}

inline String::operator const char*() const
{ 
  return str(chars());
}

inline int String::index(const Regex& r, int startpos = 0) const
{
  int unused;  return r.search(chars(), length(), unused, startpos);
}

inline  ostream& operator<<(ostream& s, const String& x)
{ 
  s.put(x.chars()); return s;
}

inline int operator==(const String& x, const String& y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const String& x, const String& y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const String& x, const String& y)
{
  return compare(x, y) > 0; 
}

inline int operator>=(const String& x, const String& y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const String& x, const String& y)
{
  return compare(x, y) < 0; 
}

inline int operator<=(const String& x, const String& y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const String& x, const SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const String& x, const SubString&  y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const String& x, const SubString&  y)      
{
  return compare(x, y) > 0; 
}

inline int operator>=(const String& x, const SubString&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const String& x, const SubString&  y) 
{
  return compare(x, y) < 0; 
}

inline int operator<=(const String& x, const SubString&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const String& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const String& x, const char* t) 
{
  return compare(x, t) != 0; 
}

inline int operator>(const String& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const String& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const String& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const String& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int operator==(const SubString& x, const String& y) 
{
  return compare(y, x) == 0; 
}

inline int operator!=(const SubString& x, const String& y)
{
  return compare(y, x) != 0;
}

inline int operator>(const SubString& x, const String& y)      
{
  return compare(y, x) < 0;
}

inline int operator>=(const SubString& x, const String& y)     
{
  return compare(y, x) <= 0;
}

inline int operator<(const SubString& x, const String& y)      
{
  return compare(y, x) > 0;
}

inline int operator<=(const SubString& x, const String& y)     
{
  return compare(y, x) >= 0;
}

inline int operator==(const SubString& x, const SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const SubString& x, const SubString&  y)
{
  return compare(x, y) != 0;
}

inline int operator>(const SubString& x, const SubString&  y)      
{
  return compare(x, y) > 0;
}

inline int operator>=(const SubString& x, const SubString&  y)
{
  return compare(x, y) >= 0;
}

inline int operator<(const SubString& x, const SubString&  y) 
{
  return compare(x, y) < 0;
}

inline int operator<=(const SubString& x, const SubString&  y)
{
  return compare(x, y) <= 0;
}

inline int operator==(const SubString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const SubString& x, const char* t) 
{
  return compare(x, t) != 0;
}

inline int operator>(const SubString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const SubString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const SubString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const SubString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int String::gsub(const String& pat, const String& r)
{
  return _gsub(pat.chars(), pat.length(), r.chars(), r.length());
}

inline int String::gsub(const SubString&  pat, const String& r)
{
  return _gsub(pat.chars(), pat.length(), r.chars(), r.length());
}

inline int String::gsub(const Regex& pat, const String& r)
{
  return _gsub(pat, r.chars(), r.length());
}

inline int String::gsub(const char* pat, const String& r)
{
  return _gsub(pat, -1, r.chars(), r.length());
}

inline int String::gsub(const char* pat, const char* r)
{
  return _gsub(pat, -1, r, -1);
}

inline void String::del(const String& y, int startpos = 0)
{
  del(search(startpos, length(), y.chars(), y.length()), y.length());
}

inline void String::del(const SubString& y, int startpos = 0)
{
  del(search(startpos, length(), y.chars(), y.length()), y.length());
}

inline void String::del(char c, int startpos = 0)
{
  del(search(startpos, length(), c), 1);
}

inline Regex::Regex(const String& x, int fast = 0, int bufsize = 40, 
             const char* transtable = 0)
{
  initialize(x.chars(), x.length(), fast, bufsize, transtable);
}

inline Regex::Regex(const char* t, int fast = 0, int bufsize = 40, 
             const char* transtable = 0)
{
  initialize(t, -1, fast, bufsize, transtable);
}

inline SubString String::_substr(int first, int l)
{
  if (first < 0 || (unsigned)(first + l) > length()) 
    return SubString(_nilString, 0, 0) ;
  else 
    return SubString(*this, first, l);
}

//#endif

#endif
