// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    based on code by Marc Shapiro (shapiro@sor.inria.fr)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#include "<T>.RPlex.h"

typedef <T>IChunk* _<T>IChunk_ptr;

<T>RPlex:: <T>RPlex()
{
  mods = 0;
  lo = fnc = 0;
  csize = DEFAULT_INITIAL_CAPACITY;
  <T>* data = new <T>[csize];
  hd = ch = new <T>IChunk(data,  lo, lo, fnc, lo+csize);
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _<T>IChunk_ptr[maxch];
  chunks[lch] = ch;
}

<T>RPlex:: <T>RPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  mods = 0;
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  lo, lo, fnc, csize+lo);
  }
  else
  {
    csize = -chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  chunksize+lo, lo, fnc, fnc);
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _<T>IChunk_ptr[maxch];
  chunks[lch] = ch;
}


<T>RPlex:: <T>RPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  mods = 0;
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  lo, lo, fnc, csize);
  }
  else
  {
    csize = -chunksize;
    <T>* data = new <T>[csize];
    hd = ch = new <T>IChunk(data,  chunksize, lo, fnc, fnc);
  }
  maxch = MIN_NCHUNKS;
  lch = maxch / 2;
  fch = lch + 1;
  base = ch->base_index() - lch * csize;
  chunks = new _<T>IChunk_ptr[maxch];
  chunks[lch] = ch;
}

void <T>RPlex::make_initial_chunks(int up = 1)
{
  int count = 0;
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      <T>* data = new <T> [csize];
      <T>IChunk* h = new <T>IChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      ++count;
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      <T>* data = new <T> [csize];
      <T>IChunk* h = new <T>IChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  ch = (<T>IChunk*)hd;
  
  maxch = MIN_NCHUNKS;
  if (maxch < count * 2)
    maxch = count * 2;
  chunks = new _<T>IChunk_ptr[maxch];
  lch = maxch / 3;
  fch = lch + count;
  base = ch->base_index() - csize * lch;
  int k = lch;
  do
  {
    chunks[k++] = ch;
    ch = ch->next();
  } while (ch != hd);
}

<T>RPlex:: <T>RPlex(int l, int hi, const <T&> initval, int chunksize = 0)
{
  mods = 0;
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

<T>RPlex::<T>RPlex(const <T>RPlex& a)
{
  mods = 0;
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
}

void <T>RPlex::operator= (const <T>RPlex& a)
{
  if (&a != this)
  {
    invalidate();
    record_change();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
  }
}


void <T>RPlex::cache(const <T>* p)
{
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->next();
    if (ch == old) index_error();
  }
}

int <T>RPlex::owns(Pix p)
{
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->next();
    if (ch == old) return 0;
  }
  return 1;
}


<T>* <T>RPlex::dosucc(<T>* p)
{
  if (p == 0) return 0;
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->next();
    if (ch == old) return 0;
  }
  int i = ch->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= ch->fence_index()) ch = ch->next();
  return ch->pointer_to(i);
}

<T>* <T>RPlex::dopred(<T>* p)
{
  if (p == 0) return 0;
  <T>IChunk* old = ch;
  while (!ch->actual_pointer(p))
  {
    ch = ch->prev();
    if (ch == old) return 0;
  }
  int i = ch->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < ch->low_index()) ch = ch->prev();
  return (ch->pointer_to(i));
}

int <T>RPlex::add_high(const <T&> elem)
{
  record_change();
  ch =  tl();
  if (!ch->can_grow_high())
  {
    <T>* data = new <T> [csize];
    ch = new <T>IChunk(data,  fnc, fnc, fnc,fnc+csize);
    ch->link_to_prev(tl());
    if (fch == maxch)
    {
      maxch *= 2;
      <T>IChunk** newch = new _<T>IChunk_ptr [maxch];
      bcopy(chunks, newch, fch * sizeof(_<T>IChunk_ptr));
      delete chunks;
      chunks = newch;
    }
    chunks[fch++] = ch;
  }
  *((ch-><T>IChunk::grow_high())) = elem;
  return fnc++;
}

int <T>RPlex::del_high ()
{
  if (empty()) empty_error();
  record_change();
  ch =  tl();
  ch-><T>IChunk::shrink_high();
  if (ch-><T>IChunk::empty() && !one_chunk())
  {
    <T>IChunk* pred = ch->prev();
    del_chunk(ch);
    ch = pred;
    --fch;
  }
  return --fnc - 1;
}

int <T>RPlex::add_low (const <T&> elem)
{
  record_change();
  ch =  hd;
  if (!ch->can_grow_low())
  {
    <T>* data = new <T> [csize];
    hd = new <T>IChunk(data,  lo-csize, lo, lo, lo);
    hd->link_to_next(ch);
    ch =  hd;
    if (lch == 0)
    {
      lch = maxch;
      fch += maxch;
      maxch *= 2;
      <T>IChunk** newch = new _<T>IChunk_ptr [maxch];
      bcopy(chunks, &(newch[lch]), lch * sizeof(_<T>IChunk_ptr));
      delete chunks;
      chunks = newch;
      base = ch->base_index() - (lch - 1) * csize;
    }
    chunks[--lch] = ch;
  }
  *((ch-><T>IChunk::grow_low())) = elem;
  return --lo;
}


int <T>RPlex::del_low ()
{
  if (empty()) empty_error();
  record_change();
  ch =  hd;
  ch-><T>IChunk::shrink_low();
  if (ch-><T>IChunk::empty() && !one_chunk())
  {
    hd = ch->next();
    del_chunk(ch);
    ch =  hd;
    ++lch;
  }
  return ++lo;
}

void <T>RPlex::append (const <T>Plex& a)
{
  for (int i = a.low(); i < a.fence(); a.next(i)) add_high(a[i]);
}

void <T>RPlex::prepend (const <T>Plex& a)
{
  for (int i = a.high(); i > a.ecnef(); a.prev(i)) add_low(a[i]);
}

void <T>RPlex::reverse()
{
  <T> tmp;
  int l = lo;
  int h = fnc - 1;
  ch = hd;
  <T>IChunk* hich = tl();
  while (l < h)
  {
    <T>* lptr = ch->pointer_to(l);
    <T>* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= ch->fence_index()) ch = ch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void <T>RPlex::fill(<T&> x)
{
  for (int i = lo; i < fnc; ++i) (*this)[i] = x;
}

void <T>RPlex::fill(<T&> x, int lo, int hi)
{
  for (int i = lo; i <= hi; ++i) (*this)[i] = x;
}


void <T>RPlex::clear()
{
  for (int i = lch + 1; i < fch; ++i)
    del_chunk(chunks[i]);
  fch = lch + 1;
  ch = chunks[lch];
  ch-><T>IChunk::clear(lo);
  fnc = lo;
}

int <T>RPlex::reset_low(int l)
{
  int old = lo;
  int diff = l - lo;
  if (diff != 0)
  {
    record_change();
    lo += diff;
    fnc += diff;
    <T>IChunk* t = hd;
    do
    {
      t->re_index(t->low_index() + diff);
      t = t->next();
    } while (t != hd);
  }
  base = hd->base_index() - lch * csize;
  return old;
}


int <T>RPlex::OK ()
{
  int v = hd != 0 && ch != 0;         // at least one chunk

  v &= fnc == tl()->fence_index();  // last chunk fnc == plex fnc
  v &= lo == hd-><T>IChunk::low_index();    // first lo == plex lo

  v &= base == hd->base_index() - lch * csize; // base is correct;
  v &= lch < fch;
  v &= fch <= maxch;                  // within allocation;

// loop for others:

  int k = lch;                        // to cross-check nch

  int found_ch = 0;                   // to make sure ch is in list;
  <T>IChunk* t = (hd);
  for (;;)
  {
    v &= chunks[k++] == t;             // each chunk is at proper index
    if (t == ch) ++found_ch;
    v &= t-><T>IChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  v &= fch == k;
  if (!v) error("invariant failure");
  return v;
}
