// This file is part of the program FRYSK.
//
// Copyright 2005, 2006, 2007, 2008, Red Hat Inc.
//
// FRYSK is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; version 2 of the License.
//
// FRYSK is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with FRYSK; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
// 
// In addition, as a special exception, Red Hat, Inc. gives You the
// additional right to link the code of FRYSK with code not covered
// under the GNU General Public License ("Non-GPL Code") and to
// distribute linked combinations including the two, subject to the
// limitations in this paragraph. Non-GPL Code permitted under this
// exception must only link to the code of FRYSK through those well
// defined interfaces identified in the file named EXCEPTION found in
// the source code files (the "Approved Interfaces"). The files of
// Non-GPL Code may instantiate templates or use macros or inline
// functions from the Approved Interfaces without causing the
// resulting work to be covered by the GNU General Public
// License. Only Red Hat, Inc. may make changes or additions to the
// list of Approved Interfaces. You must obey the GNU General Public
// License in all respects for all of the FRYSK code and other code
// used in conjunction with FRYSK except the Non-GPL Code covered by
// this exception. If you modify this file, you may extend this
// exception to your version of the file, but you are not obligated to
// do so. If you do not wish to provide this exception without
// modification, you must delete this exception statement from your
// version and license this file solely under the GPL without
// exception.

package frysk.proc.live;

import java.util.Collection;
import java.util.LinkedList;
import java.util.HashSet;
import java.util.Set;
import frysk.proc.Action;
import frysk.sys.proc.Exe;
import frysk.proc.Proc;
import frysk.proc.Task;
import frysk.proc.Host;
import frysk.sys.proc.Stat;
import frysk.proc.Auxv;
import frysk.sys.proc.AuxvBuilder;
import frysk.proc.MemoryMap;
import java.util.ArrayList;
import frysk.sys.proc.CmdLineBuilder;
import frysk.sys.proc.MapsBuilder;
import frysk.sys.ProcessIdentifier;
import frysk.sys.ProcessIdentifierFactory;
import frysk.sys.proc.Status;
import frysk.rsl.Log;
import frysk.sys.proc.ProcBuilder;
import java.util.HashMap;
import java.util.Iterator;
import java.io.File;
import frysk.proc.Manager;
import frysk.proc.ProcEvent;
import frysk.proc.TaskObserver;
import frysk.sysroot.SysRootCache;
import frysk.sysroot.SysRootFile;

/**
 * A Linux Proc tracked using PTRACE.
 */

public class LinuxPtraceProc extends LiveProc {
    private static final Log fine = Log.fine(LinuxPtraceProc.class);

    /**
     * Create a new detached process.  RUNNING makes no sense here.
     * Since PARENT could be NULL, also explicitly pass in the host.
     */
    public LinuxPtraceProc(Host host, Proc parent,
			   ProcessIdentifier pid, Stat stat) {
	super(host, parent, pid);
	this.newState = LinuxPtraceProcState.initial(false);
	this.stat = stat;
	this.breakpoints = new BreakpointAddresses(this);
	this.watchpoints = new WatchpointAddresses(this);
	((LinuxPtraceHost)host).addProc(this);
    }

    /**
     * Create a new, definitely attached, definitely running fork of
     * Task.
     */
    public LinuxPtraceProc(Task task, ProcessIdentifier fork) {
	super(task, fork);
	this.newState = LinuxPtraceProcState.initial(true);
	LinuxPtraceProc forkingProc = (LinuxPtraceProc) task.getProc();
	this.breakpoints = forkingProc.breakpoints.cloneForProc(this);
	this.watchpoints = new WatchpointAddresses(this);
	((LinuxPtraceHost)getHost()).addProc(this);
    }

    void addTask(LinuxPtraceTask task) {
	tasks.put(task.tid, task);
    }
    void removeTask(LinuxPtraceTask task) {
	tasks.remove(task.tid);
	remove(task);
    }
    Set getAllTasks() {
	return new HashSet(tasks.values());
    }
    private final HashMap tasks = new HashMap();

    public Task getMainTask() {
	if (mainTask == null) {
	    if (tasks.size() == 0) {
		// Detached, make up a main task.
		mainTask = new LinuxPtraceTask(this, pid);
	    } else {
		// Hopefully attached, extract the main task.
		mainTask = (LinuxPtraceTask)tasks.get(pid);
	    }
	}
	return mainTask;
    }
    private LinuxPtraceTask mainTask;

    public Auxv[] getAuxv() {
	if (auxv == null) {
	    class BuildAuxv extends AuxvBuilder {
		Auxv[] vec;
		public void buildDimensions (int wordSize, boolean bigEndian,
					     int length) {
		    vec = new Auxv[length];
		}
		public void buildAuxiliary (int index, int type, long val) {
		    vec[index] = new Auxv (type, val);
		}
	    }
	    BuildAuxv auxv = new BuildAuxv ();
	    ProcessIdentifier pid
		= ProcessIdentifierFactory.create(getPid());
	    auxv.construct(pid);
	    this.auxv = auxv.vec;
	}
	return auxv;
    }
    private Auxv[] auxv;

    private MemoryMap[] maps;

    public MemoryMap[] getMaps() {

	class BuildMaps extends MapsBuilder {

	    ArrayList mapsList = new ArrayList();
	    byte[] mapsLocalArray;

	    public void buildBuffer(byte[] mapsArray) {
		mapsLocalArray = mapsArray;
		mapsArray[mapsArray.length - 1] = 0;
	    }

	    public void buildMap(long addressLow, long addressHigh,
		    boolean permRead, boolean permWrite, boolean permExecute,
		    boolean shared, long offset, int devMajor, int devMinor,
		    int inode, int pathnameOffset, int pathnameLength) {
	
		byte[] mapFilename = new byte[pathnameLength];
		System.arraycopy(mapsLocalArray, pathnameOffset, mapFilename,
				 0, pathnameLength);
		
		MemoryMap map = new MemoryMap(addressLow, addressHigh,
			permRead, permWrite, permExecute, shared, offset,
			devMajor, devMinor, inode, new String(mapFilename));
		mapsList.add(map);
	    }
	}
	
	BuildMaps constructedMaps = new BuildMaps();
	constructedMaps.construct(ProcessIdentifierFactory.create(getPid()));
	MemoryMap arrayMaps[] = new MemoryMap[constructedMaps.mapsList.size()];
	constructedMaps.mapsList.toArray(arrayMaps);
	this.maps = arrayMaps;

	return maps;
    }

    private String[] cmdLine;
    public String[] getCmdLine() {
	if (cmdLine == null) {
	    class BuildCmdLine extends CmdLineBuilder {
		String[] argv;
		public void buildArgv (String[] argv) {
		    this.argv = argv;
		}
	    }
	    BuildCmdLine cmdLine = new BuildCmdLine ();
	    cmdLine.construct(ProcessIdentifierFactory.create(getPid()));
	    this.cmdLine = cmdLine.argv;
	}
	return cmdLine;
    }

    public int getUID() {
	Status status = new Status();
	status.scan(ProcessIdentifierFactory.create(getPid()));
	return status.uid;
    }

    public int getGID() {
	Status status = new Status();
	status.scan(ProcessIdentifierFactory.create(getPid()));
	return status.gid;
    }

    /**
     * XXX: This is racy - it can miss file renames.  The alternative
     * would be to have two methods; one returning a file descriptor
     * and a second returning the exe as it should be (but possibly
     * isn't :-).  Better yet have utrace handle it :-)
     */
    private String exe;
    private String getExe() {
	if (exe == null) {
	    ProcessIdentifier pid
		= ProcessIdentifierFactory.create(getPid());
	    String exe = Exe.get(pid);
	    // Linux's /proc/$$/exe can get screwed up in several
	    // ways.  Detect each here and return null.
	    if (exe.endsWith(" (deleted)"))
		// Assume (possibly incorrectly) that a trailing
		// "(deleted)" always indicates a deleted file.
		return null;
	    if (exe.indexOf((char)0) >= 0)
		// Assume that an EXE that has somehow ended up with
		// an embedded NUL character is invalid.  This happens
		// when the kernel screws up "mv a-really-long-file
		// $exe" leaving the updated EXE string with something
		// like "$exe<NUL>ally-long-file (deleted)".
		return null;
	    if (!new File(exe).exists())
		// Final sanity check; the above two should have covered
		// all possible cases.  But one never knows.
		return null;
	    this.exe = exe;
	}
	return exe;
    }

    public SysRootFile getExeFile() {
	String exe = getExe();
	File exeFile = new File(exe);
	return new SysRootFile(SysRootCache.getSysRoot(exeFile.getName()), exeFile);
    }

    /**
     * If it hasn't already been read, read the stat structure.
     */
    public Stat getStat() {
	if (stat == null) {
	    stat = new Stat().scan(ProcessIdentifierFactory.create(getPid()));
	    fine.log(this, "getStat", stat);
	}
	return stat;
    }
    private Stat stat;

    public String getCommand() {
	return getStat().comm;
    }

    /**
     * Refresh the Proc.
     */
    public void sendRefresh() {
	// Compare this against the existing TASKS.  ADDED accumulates
	// any tasks added to the TASKS.  REMOVED, starting with all
	// known tasks has any existing tasks removed, so that by the
	// end it contains a set of removed tasks.
	class TidBuilder extends ProcBuilder {
	    HashMap added = new HashMap();
	    HashMap removed = (HashMap) tasks.clone();
	    public void build(ProcessIdentifier tid) {
		if (removed.containsKey(tid)) {
		    removed.remove(tid);
		} else {
		    // Add the process (it currently isn't attached).
		    Task newTask
			= new LinuxPtraceTask(LinuxPtraceProc.this, tid);
		    added.put(tid, newTask);
		}
	    }
	}
	TidBuilder tasks = new TidBuilder ();
	tasks.construct(ProcessIdentifierFactory.create(getPid()));
	// Tell each task that no longer exists that it has been
	// removed.
	for (Iterator i = tasks.removed.values().iterator(); i.hasNext();) {
	    LinuxPtraceTask task = (LinuxPtraceTask) i.next ();
	    // XXX: Should there be a TaskEvent.schedule(), instead of
	    // Manager .eventLoop .appendEvent for injecting the event
	    // into the event loop?
	    task.performRemoval ();
	    removeTask(task);
	}
    }


    /**
     * The current state of this Proc, during a state transition
     * newState is null.
     */
    private LinuxPtraceProcState oldState;
    private LinuxPtraceProcState newState;

    /**
     * Return the current state as a string.
     */
    protected String getStateFIXME() {
	if (newState != null)
	    return newState.toString();
	else if (oldState != null)
	    return oldState.toString();
	else
	    return "<null>";
    }

    /**
     * Return the current state while at the same time marking that
     * the state is in flux. If a second attempt to change state
     * occurs before the current state transition has completed,
     * barf. XXX: Bit of a hack, but at least this prevents state
     * transition code attempting a second recursive state transition.
     */
    private LinuxPtraceProcState oldState() {
	if (newState == null)
	    throw new RuntimeException(this + " double state transition");
	oldState = newState;
	newState = null;
	return oldState;
    }
  
    /**
     * (Internal) Tell the process that is no longer listed in the
     * system table remove itself.
     */
    void performRemoval() {
	fine.log(this, "performRemoval -- no longer in /proc");
	Manager.eventLoop.add(new ProcEvent() {
		public void execute() {
		    fine.log(LinuxPtraceProc.this, "execute - performRemoval");
		    newState = oldState().handleRemoval(LinuxPtraceProc.this);
		}
	    });
    }

    /**
     *(Internal) Tell the process that the corresponding task has
     * completed its attach.
     *
     * XXX: Should not be public.
     */
    void performTaskAttachCompleted (final Task theTask) {
	fine.log(this, "performTaskAttachCompleted");
	Manager.eventLoop.add(new ProcEvent() {
		Task task = theTask;

		public void execute() {
		    newState = oldState().handleTaskAttachCompleted(LinuxPtraceProc.this, (LinuxPtraceTask) task);
		}
	    });
    }

    /**
     * (Internal) Tell the process that the corresponding task has
     * completed its detach.
     *
     * XXX: Should not be public.
     */
    void performTaskDetachCompleted(final Task theTask) {
	fine.log(this, "performTaskDetachCompleted");
	Manager.eventLoop.add(new ProcEvent() {
		Task task = theTask;
		public void execute() {
		    newState = oldState().handleTaskDetachCompleted(LinuxPtraceProc.this, (LinuxPtraceTask) task);
		}
	    });
    }

    /**
     * (Internal) Tell the process that the corresponding task has
     * completed its detach.
     */
    void performTaskDetachCompleted(final Task theTask, final Task theClone) {
	fine.log(this, "performTaskDetachCompleted/clone");
	Manager.eventLoop.add(new ProcEvent() {
		Task task = theTask;

		Task clone = theClone;

		public void execute() {
		    newState = oldState().handleTaskDetachCompleted(LinuxPtraceProc.this, (LinuxPtraceTask) task, (LinuxPtraceTask) clone);
		}
	    });
    }

    protected void performDetach() {
	fine.log(this, "performDetach");
	Manager.eventLoop.add(new ProcEvent() {
		public void execute() {
		    newState = oldState().handleDetach(LinuxPtraceProc.this, true);
		}
	    });
    }

    /**
     * (internal) Tell the process to add the specified Observation,
     * attaching the process if necessary.
     */
    void handleAddObservation(TaskObservation observation) {
	newState = oldState().handleAddObservation(this, observation);
    }

    /**
     * (Internal) Tell the process to add the specified Observation,
     * attaching to the process if necessary.
     */
    void requestAddObserver(Task task, TaskObservable observable,
			    TaskObserver observer) {
	fine.log(this, "requestAddObservation");
	Manager.eventLoop.add(new TaskObservation((LinuxPtraceTask)task,
						  observable, observer, true) {
		public void execute() {
		    handleAddObservation(this);
		}
	    });
    }

    /**
     * (Internal) Tell the process to delete the specified
     * Observation, detaching from the process if necessary. Removes a
     * syscallObserver exiting the task from syscall tracing mode of
     * necessary.
     */
    void requestDeleteObserver(Task task, TaskObservable observable,
			       TaskObserver observer) {
	Manager.eventLoop.add(new TaskObservation((LinuxPtraceTask)task,
						  observable, observer, false) {
		public void execute() {
		    newState = oldState().handleDeleteObservation(LinuxPtraceProc.this, this);
		}
	    });
    }

    /**
     * Class describing the action to take on the suspended Task
     * before adding or deleting a Syscall observer.
     */
    final class SyscallAction implements Runnable {
	private final LinuxPtraceTask task;
	private final boolean addition;
	SyscallAction(LinuxPtraceTask task, boolean addition) {
	    this.task = task;
	    this.addition = addition;
	}
	public void run() {
	    int syscallobs = task.syscallObservers.numberOfObservers();
	    if (addition) {
		if (syscallobs == 0)
		    task.startTracingSyscalls();
	    } else {
		if (syscallobs == 0)
		    task.stopTracingSyscalls();
	    }
	}
    }

    /**
     * (Internal) Tell the process to add the specified Observation,
     * attaching to the process if necessary. Adds a syscallObserver
     * which changes the task to syscall tracing mode of necessary.
     */
    void requestAddSyscallObserver(final Task task, TaskObservable observable,
				   TaskObserver observer) {
	fine.log(this, "requestAddSyscallObserver");
	SyscallAction sa = new SyscallAction((LinuxPtraceTask)task, true);
	TaskObservation to = new TaskObservation((LinuxPtraceTask)task,
						 observable, observer, sa,
						 true) {
		public void execute() {
		    handleAddObservation(this);
		}
		public boolean needsSuspendedAction() {
		    return ((LinuxPtraceTask)task).syscallObservers.numberOfObservers() == 0;
		}
	    };
	Manager.eventLoop.add(to);
    }

    /**
     * (Internal) Tell the process to delete the specified
     * Observation, detaching from the process if necessary.
     */
    void requestDeleteSyscallObserver(final Task task,
				      TaskObservable observable,
				      TaskObserver observer) {
	fine.log(this, "requestDeleteSyscallObserver");
	SyscallAction sa = new SyscallAction((LinuxPtraceTask)task, false);
	TaskObservation to = new TaskObservation((LinuxPtraceTask)task,
						 observable, observer, sa,
						 false) {
		public void execute() {
		    newState = oldState().handleDeleteObservation(LinuxPtraceProc.this,
								  this);
		}

		public boolean needsSuspendedAction() {
		    return ((LinuxPtraceTask)task).syscallObservers.numberOfObservers() == 1;
		}
	    };
	Manager.eventLoop.add(to);
    }

    /**
     * Class describing the action to take on the suspended Task
     * before adding or deleting a Code observer.
     */
    final class BreakpointAction implements Runnable {
	private final TaskObserver.Code code;

	private final Task task;

	private final long address;

	private final boolean addition;

	BreakpointAction(TaskObserver.Code code, Task task, long address,
			 boolean addition) {
	    this.code = code;
	    this.task = task;
	    this.address = address;
	    this.addition = addition;
	}

	public void run() {
	    if (addition) {
	        BreakpointAddresses.CodeObserver observer;
		boolean mustInstall;
		observer = new BreakpointAddresses.CodeObserver(task, code);
		mustInstall = breakpoints.addBreakpoint(observer, address);
		if (mustInstall) {
		    Breakpoint breakpoint;
		    breakpoint = Breakpoint.create(address, LinuxPtraceProc.this);
		    breakpoint.install(task);
		}
	    } else {
	        BreakpointAddresses.CodeObserver observer;
		boolean mustRemove;
		observer = new BreakpointAddresses.CodeObserver(task, code);
		mustRemove = breakpoints.removeBreakpoint(observer, address);
		if (mustRemove) {
		    Breakpoint breakpoint;
		    breakpoint = Breakpoint.create(address, LinuxPtraceProc.this);
		    breakpoint.remove(task);
		}
	    }
	}
    }

    /**
     * (Internal) Tell the process to add the specified Code
     * Observation, attaching to the process if necessary. Adds a
     * TaskCodeObservation to the eventloop which instructs the task
     * to install the breakpoint if necessary.
     */
    void requestAddCodeObserver(Task task, TaskObservable observable,
				TaskObserver.Code observer,
				final long address) {
	fine.log(this, "requestAddCodeObserver");
	BreakpointAction bpa = new BreakpointAction(observer, task, address, true);
	TaskObservation to;
	to = new TaskObservation((LinuxPtraceTask) task, observable, observer,
				 bpa, true) {
		public void execute() {
		    handleAddObservation(this);
		}
		public boolean needsSuspendedAction() {
		    return breakpoints.getCodeObservers(address) == null;
		}
	    };
	Manager.eventLoop.add(to);
    }

    /**
     * (Internal) Tell the process to delete the specified Code
     * Observation, detaching from the process if necessary.
     */
    void requestDeleteCodeObserver(Task task, TaskObservable observable,
				   TaskObserver.Code observer,
				   final long address)    {
	fine.log(this, "requestDeleteCodeObserver");
	BreakpointAction bpa = new BreakpointAction(observer, task, address, false);
	TaskObservation to;
	to = new TaskObservation((LinuxPtraceTask)task, observable, observer, bpa, false) {
		public void execute() {
		    newState = oldState().handleDeleteObservation(LinuxPtraceProc.this, this);
		}

		public boolean needsSuspendedAction() {
		    return breakpoints.getCodeObservers(address).size() == 1;
		}
	    };

	Manager.eventLoop.add(to);
    }

    

    /**
     * Watchpoint action class. Describes the action to 
     * take on a blocked Task before adding or deleting 
     * a Watchpoint observer.
     */
    final class WatchpointAction implements Runnable {
	private final TaskObserver.Watch watch;

	private final Task task;
	private final long address;
	private final int length;
	private final boolean writeOnly;
	private final boolean addition;

	WatchpointAction(TaskObserver.Watch watch, Task task, long address,
			 int length, boolean writeOnly, boolean addition) {
	    this.watch = watch;
	    this.task = task;
	    this.address = address;
	    this.addition = addition;
	    this.length = length;
	    this.writeOnly = writeOnly;

	}

	public void run() {
	    if (addition) {		
		boolean mustInstall = watchpoints.addWatchpoint(watch, task, address, length, writeOnly);		
		if (mustInstall) {
		    Watchpoint watchpoint;
		    watchpoint = Watchpoint.create(address, length, writeOnly, task);
		    watchpoint.install(task);
		}
	    } else {
		boolean mustRemove = watchpoints.removeWatchpoint(watch, task, address, length, writeOnly);
		if (mustRemove) {
		    Watchpoint watchpoint;
		    watchpoint = Watchpoint.create(address, length, writeOnly, task);
		    watchpoint.remove(task);
		}
	    }
	}
    }

    // List and manager for Watchpoints within a task.
    protected final WatchpointAddresses watchpoints;
    
    /**
     * (Internal) Tell the process to add the specified Watch
     * Observation, attaching to the process if necessary. Adds a
     * TaskWatchObservation to the eventloop which instructs the task
     * to install the watchpoint if necessary.
     */
    void requestAddWatchObserver(final Task task, TaskObservable observable,
				TaskObserver.Watch observer,
				final long address,
				final int length,
				final boolean writeOnly) {
	fine.log(this, "requestAddWatchObserver");
	WatchpointAction wpa = new WatchpointAction(observer, task, address, length, writeOnly, true);
	TaskObservation to;
	to = new TaskObservation((LinuxPtraceTask) task, observable, observer, wpa, true) {	    	    
	    public void execute() {
		handleAddObservation(this);
	    }
	    public boolean needsSuspendedAction() {
		return watchpoints.getWatchObservers(task, address, length, writeOnly) == null;
	    }	    
	    public void add() {
		try {
		    super.add();
		} catch (Exception e) {
		    // On any exceptions being raised in add(), call
		    getTaskObserver().addFailed(this,e);
		}
		
	    }
	};
	Manager.eventLoop.add(to);
    }

    /**
     * (Internal) Tell the process to delete the specified Watch
     * Observation, detaching from the process if necessary.
     */
    void requestDeleteWatchObserver(final Task task, TaskObservable observable,
				   TaskObserver.Watch observer,
				   final long address,
				   final int length,
				   final boolean writeOnly)    {
	
	fine.log(this, "requestDeleteWatchObserver");
	WatchpointAction wpa = new WatchpointAction(observer, task, address, length, writeOnly, false);
	TaskObservation to;
	to = new TaskObservation((LinuxPtraceTask)task, observable, observer, wpa, false) {
	    public void execute() {
		newState = oldState().handleDeleteObservation(LinuxPtraceProc.this, this);
	    }
	    public boolean needsSuspendedAction() {
		return watchpoints.getWatchObservers(task, address, length, writeOnly).size() == 1;
	    }
	};
	Manager.eventLoop.add(to);
    }

    /**
     * Class describing the action to take on the suspended Task
     * before adding or deleting an Instruction observer. No
     * particular actions are needed, but we must make sure the Task
     * is suspended.
     */
    final static class InstructionAction implements Runnable {
	public void run()
	{
	    // There is nothing in particular we need to do. We just want
	    // to make sure the Task is stopped so we can send it a step
	    // instruction or, when deleted, start resuming the process
	    // normally.

	    // We do want an explicit updateExecuted() call, after adding
	    // the observer, but while still suspended. This is done by
	    // overriding the add() method in the TaskObservation
	    // below. No such action is required on deletion.
	}
    }

    /**
     * (Internal) Tell the process to add the specified Instruction
     * Observation, attaching and/or suspending the process if
     * necessary. As soon as the observation is added and the task
     * isn't blocked it will inform the Instruction observer of every
     * step of the task.
     */
    void requestAddInstructionObserver(final Task task,
				       TaskObservable observable,
				       TaskObserver.Instruction observer) {
	fine.log(this, "requestAddInstructionObserver");
	TaskObservation to;
	InstructionAction ia = new InstructionAction();
	to = new TaskObservation((LinuxPtraceTask)task, observable, observer, ia, true) {
		public void execute() {
		    handleAddObservation(this);
		}

		public boolean needsSuspendedAction() {
		    return ((LinuxPtraceTask)task).instructionObservers.numberOfObservers() == 0;
		}

		// Makes sure that the observer is properly added and then,
		// while the Task is still suspended, updateExecuted() is
		// called. Giving the observer a chance to inspect and
		// possibly block the Task.
		public void add() {
		    super.add();
		    TaskObserver.Instruction i = (TaskObserver.Instruction) observer;
		    if (i.updateExecuted(task) == Action.BLOCK)
			((LinuxPtraceTask)task).blockers.add(observer);
		}
	    };
	Manager.eventLoop.add(to);
    }

    /**
     * (Internal) Tell the process to delete the specified Instruction
     * Observation, detaching and/or suspending from the process if
     * necessary.
     */
    void requestDeleteInstructionObserver(final Task task,
					  TaskObservable observable,
					  TaskObserver.Instruction observer) {
	fine.log(this, "requestDeleteInstructionObserver");
	TaskObservation to;
	InstructionAction ia = new InstructionAction();
	to = new TaskObservation((LinuxPtraceTask)task, observable, observer, ia, false) {
		public void execute() {
		    newState = oldState().handleDeleteObservation(LinuxPtraceProc.this, this);
		}
		public boolean needsSuspendedAction() {
		    return ((LinuxPtraceTask)task).instructionObservers.numberOfObservers() == 1;
		}
	    };
	Manager.eventLoop.add(to);
    }


    /**
     * The set of observations that currently apply to this task.
     * Note that this is a Collection that may contain the same
     * Observer object multiple times (for possible different
     * observations).
     */
    private final Collection observations = new LinkedList();

    boolean addObservation(Object o) {
	return observations.add(o);
    }

    boolean removeObservation(Object o) {
	return observations.remove(o);
    }

    int observationsSize() {
	return observations.size();
    }

    Iterator observationsIterator() {
	return observations.iterator();
    }

    void requestUnblock(TaskObserver observerArg) {
	Iterator iter = getTasks().iterator();
	while (iter.hasNext()) {
	    Task task =(Task) iter.next();
	    task.requestUnblock(observerArg);
	}
    }

    /**
     * The addresses on which we have installed breakpoints (with
     * associated Tasks and TaskObserver.Code listeners). Package
     * private for access in LinuxPtraceTask constructor that creates
     * a forked task and the LinuxPtraceTaskState.MainTask.wantToDetach
     * class to clearthem. See the comments there.
     */
    final BreakpointAddresses breakpoints;
    
    // List of available addresses for out of line stepping.
    // Used a lock in getOutOfLineAddress() and doneOutOfLine().
    private final ArrayList outOfLineAddresses = new ArrayList();

    // Whether the Isa has been asked for addresses yet.
    // Guarded by outOfLineAddresses in getOutOfLineAddress.
    private boolean requestedOutOfLineAddresses;

    /**
     * Returns an available address for out of line stepping. Blocks
     * till an address is available. Queries the Isa if not done so
     * before.  Returned addresses should be returned by calling
     * doneOutOfLine().
     */
    long getOutOfLineAddress() {
	synchronized (outOfLineAddresses) {
	    while (outOfLineAddresses.isEmpty()) {
		if (! requestedOutOfLineAddresses) {
		    Isa isa = ((LinuxPtraceTask)getMainTask()).getIsaFIXME();
		    outOfLineAddresses.addAll(isa.getOutOfLineAddresses(this));
		    if (outOfLineAddresses.isEmpty())
			throw new IllegalStateException("Isa.getOutOfLineAddresses"
							+ " returned empty List");
		    requestedOutOfLineAddresses = true;
		} else {
		    try {
			outOfLineAddresses.wait();
		    } catch (InterruptedException ignored) {
			// Just try again...
		    }
		}
	    }
	    return ((Long) outOfLineAddresses.remove(0)).longValue();
	}
    }

    /**
     * Called by Breakpoint with an address returned by
     * getOutOfLineAddress() to put it back in the pool.
     */
    void doneOutOfLine(long address) {
	synchronized (outOfLineAddresses) {
	    outOfLineAddresses.add(Long.valueOf(address));
	    outOfLineAddresses.notifyAll();
	}
    }
}
