//=============================================================================
//
//      plf_stub.c
//
//      Platform specific code for GDB stub support.
//
// Modification : adaptation to EVMCAN (not tested)
//=============================================================================


#include <pkgconf/hal.h>

#ifdef CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS

#include <cyg/hal/hal_stub.h>

#include <cyg/hal/hal_io.h>             // HAL IO macros
#ifdef CYGDBG_HAL_DEBUG_GDB_BREAK_SUPPORT
#include <cyg/hal/hal_intr.h>           // HAL interrupt macros
#include <cyg/hal/drv_api.h>            // HAL ISR support
#endif

//-----------------------------------------------------------------------------
// Serial definitions.

#if CYGHWR_HAL_ARM_EVMCAN_GDB_BAUD==9600
#define CYG_DEVICE_SERIAL_RS232_BAUD        213
#endif
#if CYGHWR_HAL_ARM_EVMCAN_GDB_BAUD==19200
#define CYG_DEVICE_SERIAL_RS232_BAUD        107
#endif
#if CYGHWR_HAL_ARM_EVMCAN_GDB_BAUD==38400
#define CYG_DEVICE_SERIAL_RS232_BAUD        53
#endif
#if CYGHWR_HAL_ARM_EVMCAN_GDB_BAUD==115200
#define CYG_DEVICE_SERIAL_RS232_BAUD        18
#endif
/****************************************************************************
* USART Module structure
****************************************************************************/
typedef struct
{
    cyg_uint32  PER;               /* PIO Enable Register            */
    cyg_uint32  PDR;               /* PIO Disable Register           */
    cyg_uint32  PSR;               /* PIO Status Register            */
    cyg_uint32  ReservedA;
    cyg_uint32  OER;               /* Output Enable Register         */
    cyg_uint32  ODR;               /* Output Disable Register        */
    cyg_uint32  OSR;               /* Output Status Register         */
    cyg_uint32  ReservedB[5];
    cyg_uint32  SODR;              /* Set Output Data Register       */
    cyg_uint32  CODR;              /* Clear Output Data Register     */
    cyg_uint32  ODSR;              /* Output Data Status Register    */ 
    cyg_uint32  PDSR;              /* Pin Data Status Register       */
    cyg_uint32  MDER;              /* Multi-Driver Enable Register   */
    cyg_uint32  MDDR;              /* Multi-Driver Disable Register  */
    cyg_uint32  MDSR;              /* Multi-Driver Status Register   */
    cyg_uint32  ReservedC;
    cyg_uint32  ECR;               /* Enable Clock Register          */
    cyg_uint32  DCR;               /* Disable Clock Register         */
    cyg_uint32  PMSR;              /* Power Managt Status Register   */
    cyg_uint32  ReservedD;
    cyg_uint32  CR;                /* Control Register               */
    cyg_uint32  MR;                /* Mode Register                  */
    cyg_uint32  ReservedE[2];
    cyg_uint32  SR;                /* Status Register                */
    cyg_uint32  IER;               /* Interrupt Enable Register      */
    cyg_uint32  IDR;               /* Interrupt Disable Register     */
    cyg_uint32  IMR;               /* Interrupt Mask Register        */
    cyg_uint32  RHR;               /* Receive Holding Register       */
    cyg_uint32  THR;               /* Transmit Holding Register      */
    cyg_uint32  BRGR;              /* Baud Rate Generator Register   */
    cyg_uint32  RTOR;              /* Receiver Timeout Register      */
    cyg_uint32  TTGR;              /* Transmitter Timeguard Register */
} EASYCAN_RTOUSART_REGS;

/****************************************************************************
* PIO controller block PER, PDR, PSR, OER, ODR, OSR, SODR, CODR, ODSR,
*                      PDSR, MDER, MDDR, MDSR, SR, IER, IDR, IMR
****************************************************************************/
#define SCK                (0x01 << 16)  // SCK
#define TXD                (0x01 << 17)  // TXD
#define RXD                (0x01 << 18)  // RXD

/****************************************************************************
* PMC controller block ECR, DCR, PMSR
****************************************************************************/
#define PIO                (0x01 << 0)   // PIO
#define USART              (0x01 << 1)   // USART

/****************************************************************************
* Control register  CR
****************************************************************************/
#define SWRST              (0x01 << 0)   // USART software reset
#define RSTRX              (0x01 << 2)   // Reset receiver
#define RSTTX              (0x01 << 3)   // Reset transmitter
#define RXEN               (0x01 << 4)   // Receiver enable
#define RXDIS              (0x01 << 5)   // Receiver disable
#define TXEN               (0x01 << 6)   // Transmitter enable
#define TXDIS              (0x01 << 7)   // Transmitter disable
#define RSTSTA             (0x01 << 8)   // Reset status bits
#define STTBRK             (0x01 << 9)   // Start break
#define STPBRK             (0x01 << 10)  // Stop break
#define STTTO              (0x01 << 11)  // Start timeout
#define SENDA              (0x01 << 12)  // Send address

/****************************************************************************
* Mode register  MR
****************************************************************************/
#define USCLKS             (0x03 << 4)   // Clock selection mask
#define USCLKS_MCKI        (0x00 << 4)   // Internal clock MCKI
#define USCLKS_MCKI_8      (0x01 << 4)   // Internal clock MCKI/8
#define USCLKS_SCK         (0x02 << 4)   // External clock

#define CHRL               (0x03 << 6)   // Byte length
#define CHRL_5             (0x00 << 6)   // Five bits
#define CHRL_6             (0x01 << 6)   // Six bits
#define CHRL_7             (0x02 << 6)   // Seven bits
#define CHRL_8             (0x03 << 6)   // Heigh bits

#define SYNC               (0x01 << 8)   // Synchronous mode
#define ASYNC              (0x00 << 8)   // Asynchronous mode

#define PAR                (0x07 << 9)   // Parity mode
#define PAR_EVEN           (0x00 << 9)   // Even parity
#define PAR_ODD            (0x01 << 9)   // Odd parity
#define PAR_SPACE          (0x02 << 9)   // Space parity (forced to 0)
#define PAR_MARK           (0x03 << 9)   // Mark parity (forced to 1)
#define PAR_NO             (0x04 << 9)   // No parity
#define PAR_MULTIDROP      (0x06 << 9)   // Multi drop parity
 
#define IDLE               (0x01 << 10)  // Idle (J1708 Protocol)
#define IDLEFLAG           (0x01 << 11)  // Idle Flag (J1708 Protocol)

#define NBSTOP             (0x03 << 12)  // Stop bit number
#define NBSTOP_1           (0x00 << 12)  // 1 stop bit
#define NBSTOP_15          (0x01 << 12)  // 1.5 stop bit
#define NBSTOP_2           (0x02 << 12)  // 2 stop bit

#define CHMODE             (0x03 << 14)  // Channel mode
#define CHMODE_NORMAL      (0x00 << 14)  // Normal channel
#define CHMODE_AUTO        (0x01 << 14)  // Automatic echo channel
#define CHMODE_LOCAL       (0x02 << 14)  // Local loopback channel
#define CHMODE_REMOTE      (0x03 << 14)  // Remote loopback channel

#define MODE9              (0x01 << 17)  // 9 bit mode
#define MODE8              (0x00 << 17)  // 8 bit mode
#define CLKO               (0x01 << 18)  // Clock output
#define CLKI               (0x00 << 18)  // Clock input

/****************************************************************************
* Channel Status and Interrupt registers  IER, IDR, IMR, SR
****************************************************************************/
#define RXRDY              (0x01 << 0)   // Receiver ready
#define TXRDY              (0x01 << 1)   // Transmitter ready
#define RXBRK              (0x01 << 2)   // Receiver break
#define ENDRX              (0x01 << 3)   // End of receiver PDC transfer
#define ENDTX              (0x01 << 4)   // End of transmitter PDC transfer
#define USOVRE             (0x01 << 5)   // Overrun error
#define FRAME              (0x01 << 6)   // Framing error
#define PARE               (0x01 << 7)   // Parity error
#define TIMEOUT            (0x01 << 8)   // Receiver time out
#define TXEMPTY            (0x01 << 9)   // Transmitter empty

/****************************************************************************
* Time out 
****************************************************************************/
#define TRANSMIT_TIMEOUT   0xFF
#define RECEIVE_TIMEOUT    0xFF
#define US_BDR_9600        1



#if CYGHWR_HAL_ARM_EVMCAN_GDB_PORT==0
#define EASYCAN_RTOUSART_BASE        0xFFFA8000
#define CYG_DEVICE_SERIAL_INT  CYGNUM_HAL_INTERRUPT_USART0
#else 
 #if CYGHWR_HAL_ARM_EVMCAN_GDB_PORT==1
 #define EASYCAN_RTOUSART_BASE        0xFFFAC000
 #define CYG_DEVICE_SERIAL_INT  CYGNUM_HAL_INTERRUPT_USART1
 #else
 #define EASYCAN_RTOUSART_BASE        0xFFFB0000
 #define CYG_DEVICE_SERIAL_INT  CYGNUM_HAL_INTERRUPT_USART2
 #endif
#endif
#ifdef CYGDBG_HAL_DEBUG_GDB_BREAK_SUPPORT
static cyg_interrupt gdb_interrupt;
static cyg_handle_t  gdb_interrupt_handle;
#endif

//-----------------------------------------------------------------------------
#ifdef CYGDBG_HAL_DEBUG_GDB_BREAK_SUPPORT
// This ISR is called only for serial receive interrupts.
int 
cyg_hal_gdb_isr(cyg_vector_t vector, cyg_addrword_t data, HAL_SavedRegisters *regs)
{
    cyg_uint8 c;

    c = hal_evmcan_get_char();  // Fetch the character
    if( 3 == c ) {  // ^C
        // Ctrl-C: set a breakpoint at PC so GDB will display the
        // correct program context when stopping rather than the
        // interrupt handler.
        cyg_hal_gdb_interrupt (regs->pc);
        cyg_drv_interrupt_acknowledge(CYG_DEVICE_SERIAL_INT);
    }
    return 0;  // No need to run DSR
}

int 
hal_evmcan_interruptible(int state)
{
    if (state) {
        cyg_drv_interrupt_unmask(CYG_DEVICE_SERIAL_INT);
    } else {
        cyg_drv_interrupt_mask(CYG_DEVICE_SERIAL_INT);
    }

    return 0;
}
#endif

// Initialize the current serial port.
void hal_evmcan_init_serial(void)
{
    int loop_count;
    volatile struct EASYCAN_RTOUSART_REGS *RTOusart =
                            (volatile struct EASYCAN_RTOUSART_REGS*)EASYCAN_RTOUSART_BASE;

  /* Switch On USART */
  RTOusart->ECR = (USART | PIO);

  /* Reset Error Flags Rx & Tx State Machine */
  RTOusart->CR = (RSTRX | RSTTX);
  RTOusart->CR =  RSTSTA;
  
  /* Set Mode */
  RTOusart->MR =CHRL_8|PAR_NO|NBSTOP_1|USCLKS_MCKI;

 


    // Set baud rate.
     RTOusart->BRGR = CYG_DEVICE_SERIAL_RS232_BAUD;
     
    /* Set Rx & Tx Line as Peripheral Lines */
    RTOusart->PDR = (TXD | RXD);


#ifdef CYGDBG_HAL_DEBUG_GDB_BREAK_SUPPORT
    cyg_drv_interrupt_create(CYG_DEVICE_SERIAL_INT,
                             99,                     // Priority - what goes here?
                             0,                      //  Data item passed to interrupt handler
                             cyg_hal_gdb_isr,
                             0,
                             &gdb_interrupt_handle,
                             &gdb_interrupt);
    cyg_drv_interrupt_attach(gdb_interrupt_handle);
    cyg_drv_interrupt_unmask(CYG_DEVICE_SERIAL_INT);
    /* enable receive interupt */
    RTOusart->IER = RXRDY;
#endif
    

    /* enable Tx and Rx */
    RTOusart->CR = TXEN|RXEN;
    for (loop_count = 1000 ; loop_count > 0 ; loop_count--);
}

// Write C to the current serial port.
void hal_evmcan_put_char(int c)
{
    volatile struct EASYCAN_RTOUSART_REGS *RTOusart =
                            (volatile struct EASYCAN_RTOUSART_REGS*)EASYCAN_RTOUSART_BASE;

    while ((RTOusart->SR & TXRDY) == 0);
    RTOusart->THR = c;
}

// Read one character from the current serial port.
int hal_evmcan_get_char(void)
{
    char c;
    volatile struct EASYCAN_RTOUSART_REGS *RTOusart =
                            (volatile struct EASYCAN_RTOUSART_REGS*)EASYCAN_RTOUSART_BASE;

    while ((RTOusart->CSR & RXRDY) == 0) ;   // Wait for char
    c = RTOusart->RHR;

    return c;
}

#endif // ifdef CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS
//-----------------------------------------------------------------------------
// End of plf_stub.c
