/*------------------------------------------------------------------------*/
// On-board timer
/*------------------------------------------------------------------------*/
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    Eric Miniere 
// Contributors: Eric Miniere 
// Date:         2001-09-00
// Purpose:      Define timer 
// Description:  The timer details for the EVMCAN/EASYCAN3 are defined here.
// Usage:
//               ...
//              
//
//####DESCRIPTIONEND####
//
//==========================================================================


#define CSP_REGISTER_T volatile unsigned int


/* define constants for type of physical timer                               */
#define TC_ONE_CHANNEL    1
#define TC_THREE_CHANNEL  3


/******************************************************************************
* TC_CVR : TC Channel Counter Value Register
* TC_RA  : TC Channel Register A
* TC_RB  : TC Channel Register B
* TC_RC  : TC Channel Register C
******************************************************************************/
#define TC_COMPARE_RA   0
#define TC_COMPARE_RB   1
#define TC_COMPARE_RC   2


/******************************************************************************
* Timer channel structure
******************************************************************************/
typedef struct
{
    CSP_REGISTER_T  PER;               /* PIO Enable Register                */
    CSP_REGISTER_T  PDR;               /* PIO Disable Register               */
    CSP_REGISTER_T  PSR;               /* PIO Status Register                */
    CSP_REGISTER_T  ReservedA;
    CSP_REGISTER_T  OER;               /* Output Enable Register             */
    CSP_REGISTER_T  ODR;               /* Output Disable Register            */
    CSP_REGISTER_T  OSR;               /* Output Status Register             */
    CSP_REGISTER_T  ReservedB[5];
    CSP_REGISTER_T  SODR;              /* Set Output Data Register           */
    CSP_REGISTER_T  CODR;              /* Clear Output Data Register         */
    CSP_REGISTER_T  ODSR;              /* Output Data Status Register        */
    CSP_REGISTER_T  PDSR;              /* Pin Data Status Register           */
    CSP_REGISTER_T  MDER;              /* Multi-Driver Enable Register       */
    CSP_REGISTER_T  MDDR;              /* Multi-Driver Disable Register      */
    CSP_REGISTER_T  MDSR;              /* Multi-Driver Status Register       */
    CSP_REGISTER_T  ReservedC;
    CSP_REGISTER_T  ECR;               /* Enable Clock Register              */
    CSP_REGISTER_T  DCR;               /* Disable Clock Register             */
    CSP_REGISTER_T  PMSR;              /* Power Management Status Register   */
    CSP_REGISTER_T  ReservedD;
    CSP_REGISTER_T  CR;                /* Control Register                   */
    CSP_REGISTER_T  MR;                /* Mode Register                      */
    CSP_REGISTER_T  ReservedE[2];
    CSP_REGISTER_T  SR;                /* Status Register                    */
    CSP_REGISTER_T  IER;               /* Interrupt Enable Register          */
    CSP_REGISTER_T  IDR;               /* Interrupt Disable Register         */
    CSP_REGISTER_T  IMR;               /* Interrupt Mask Register            */
    CSP_REGISTER_T  CV;                /* Counter value                      */
    CSP_REGISTER_T  RA;                /* Register A                         */
    CSP_REGISTER_T  RB;                /* Register B                         */
    CSP_REGISTER_T  RC;                /* Register C                         */
    CSP_REGISTER_T  ReservedF[28];
} CSP_RTOTC_T, *CSP_RTOTC_PTR;


/****************************************************************************
* Timer 1 channel structure
****************************************************************************/
typedef struct
{
    CSP_RTOTC_T         CHANNEL[TC_ONE_CHANNEL];    /* TC Channel            */
    CSP_REGISTER_T      ReservedA[192];
    CSP_REGISTER_T      TSTC;                       /* Test Control Register */
    CSP_REGISTER_T      TSTM;                       /* Test Mode Register    */
} CSP_RTOT1C_T, *CSP_RTOT1C_PTR;


/******************************************************************************
* Timer 3 channel structure
******************************************************************************/
typedef struct
{
    CSP_RTOTC_T         CHANNEL[TC_THREE_CHANNEL];  /* TC Channel            */
    CSP_REGISTER_T      BCR;                        /* Control Register      */
    CSP_REGISTER_T      BMR;                        /* Mode Register         */
    CSP_REGISTER_T      ReservedA[62];
    CSP_REGISTER_T      TSTC;                       /* Test Control Register */
    CSP_REGISTER_T      TSTM;                       /* Test Mode Register    */
} CSP_RTOT3C_T, *CSP_RTOT3C_PTR;


/******************************************************************************
* PIO controller block  PER, PDR, PSR, OER, ODR, OSR, SODR, CODR, ODSR,
                        PDSR, MDER, MDDR, MDSR, SR, IER, IDR, IMR
******************************************************************************/
#define TIOB            (0x01 << 16)  /* TIOB                                */
#define TIOA            (0x01 << 17)  /* TIOA                                */
#define TCLK            (0x01 << 18)  /* TCLK                                */
#define CPCS_int            (0x01 << 4)                                   

/******************************************************************************
* PMC controller block  ECR, DCR, PMSR
******************************************************************************/
#define PIO          (0x01 << 0)   /* PIO                                 */
#define TIMER           (0x01 << 1)   /* TIMER                               */


/******************************************************************************
* Timer Channel Control Register CR
******************************************************************************/
#define CHANNEL_SWRST   (0x01 << 0)    /* Timer software reset               */
#define CLKEN           (0x01 << 1)    /* Counter Clock Enable Command       */
#define CLKDIS          (0x01 << 2)    /* Counter Clock Disable Command      */
#define SWTRG           (0x01 << 3)    /* Software Trigger Command           */


/******************************************************************************
* Timer Channel Mode Register (capture mode)
******************************************************************************/
/* CLKS : clock source selection */
#define CLKS_MCK2       (0x00 << 0)    /* MCK/2                              */
#define CLKS_MCK8       (0x01 << 0)    /* MCK/8                              */
#define CLKS_MCK32      (0x02 << 0)    /* MCK/32                             */
#define CLKS_MCK128     (0x03 << 0)    /* MCK/128                            */
#define CLKS_MCK1024    (0x04 << 0)    /* MCK/1024                           */
#define CLKS_XC0        (0x05 << 0)    /* External XC0                       */
#define CLKS_XC1        (0x06 << 0)    /* External XC1                       */
#define CLKS_XC2        (0x07 << 0)    /* External XC2                       */
#define CLKS            (0x07 << 0)    /* CLKS mask                          */

/* CLKI : Clock invert */
#define CLKINV          (0x01 << 3)

/* BURST : Burst signal selection */
#define BURST_NONE      (0x00 << 4)    /* No signal selected for burst       */
#define BURST_XC0       (0x01 << 4)    /* XC0 selected for burst             */
#define BURST_XC1       (0x02 << 4)    /* XC1 selected for burst             */
#define BURST_XC2       (0x03 << 4)    /* XC2 selected for burst             */
#define BURST           (0x03 << 4)    /* Burst mask                         */

/* LDBSTOP : Counter clock stopped with RB Loading */
#define LDBSTOP         (0x01 << 6)

/* LDBDIS : Counter clock disable with RB Loading */
#define LDBDIS          (0x01 << 7)

/* ETRGEDG : External trigger edge selection */
#define ETRGEDG_NONE    (0x00 << 8)    /* No external trigger selected       */
#define ETRGEDG_RISING  (0x01 << 8)    /* Rising edge trigger                */
#define ETRGEDG_FALLING (0x02 << 8)    /* Falling edge trigger               */
#define ETRGEDG_EACH    (0x03 << 8)    /* Each edge trigger                  */
#define ETRGEDG         (0x03 << 8)    /* Trigger mask                       */

/* ABETRG : TIOA or TIOB external trigger selection */
#define ABETRG_TIOA     (0x01 << 10)   /* TIOA selected for external trigger */
#define ABETRG_TIOB     (0x00 << 10)   /* TIOB selected for external trigger */
#define ABETRG          (0x01 << 10)   /* External trigger selection mask    */

/* CPCTRG : RC compare trigger enable */
#define CPCTRG          (0x01 << 14)

/* WAVE = 0  : Capture mode is enabled */
#define WAVE_ENA        (0x01 << 15)   /* Waveform mode                      */
#define CAPT_ENA        (0x00 << 15)   /* Capture mode                       */
#define WAVE            (0x01 << 15)   /* Wave/Capture selection             */

/* LDRA : RA loadig selection */
#define LDRA_NONE       (0x00 << 16)   /* No external trigger selected       */
#define LDRA_RISING     (0x01 << 16)   /* Rising edge on TIOA                */
#define LDRA_FALLING    (0x02 << 16)   /* Falling edge on TIOA               */
#define LDRA_EACH       (0x03 << 16)   /* Each edge on TIOA                  */
#define LDRA            (0x03 << 16)   /* Load RA mask                       */

/* LDRB : RB loadig selection */
#define LDRB_NONE       (0x00 << 18)   /* No external trigger selected       */
#define LDRB_RISING     (0x01 << 18)   /* Rising edge on TIOA                */
#define LDRB_FALLING    (0x02 << 18)   /* Falling edge on TIOA               */
#define LDRB_EACH       (0x03 << 18)   /* Each edge on TIOA                  */
#define LDRB            (0x03 << 18)   /* Load RB mask                       */


/******************************************************************************
* Timer Channel Mode Register (waveform mode)
******************************************************************************/
/* CLKS  : Clock selection         : same as capture Mode                    */
/* CLKI  : Clock invert            : same as capture Mode                    */
/* BURST : Burst signal selection  : same as capture Mode                    */

/* CPCSTOP : Counter clock stopped with RC compare */
#define CPCSTOP_DIS     (0x00 << 6)
#define CPCSTOP_ENA     (0x01 << 6)
#define CPCSTOP         (0x01 << 6)

/* CPCDIS : Counter clock disable with RC compare */
#define CPCSDIS_DIS     (0x00 << 7)
#define CPCSDIS_ENA     (0x01 << 7)
#define CPCSDIS         (0x01 << 7)

/* EEVTEDG : External event edge selection */
#define EEVTEDG_NONE    (0x00 << 8)
#define EEVTEDG_RISING  (0x01 << 8)
#define EEVTEDG_FALLING (0x02 << 8)
#define EEVTEDG_EACH    (0x03 << 8)
#define EEVTEDG         (0x03 << 8)

/* EEVT : External event selection */
#define EEVT_TIOB       (0x00 << 10)
#define EEVT_XC0        (0x01 << 10)
#define EEVT_XC1        (0x02 << 10)
#define EEVT_XC2        (0x03 << 10)
#define EEVT            (0x03 << 10)

/* ENETRG : External event trigger enable */
#define ENETRG          (0x01 << 12)

/* CPCTRG   : RC compare trigger enable */
/* WAVE = 1 : Waveform mode is enable */

/* ACPA : RA compare effect on TIOA */
#define ACPA_NONE       (0x00 << 16)
#define ACPA_SET        (0x01 << 16)
#define ACPA_CLEAR      (0x02 << 16)
#define ACPA_TOGGLE     (0x03 << 16)
#define ACPA            (0x03 << 16)

/* ACPC : RC compare effect on TIOA */
#define ACPC_NONE       (0x00 << 18)
#define ACPC_SET        (0x01 << 18)
#define ACPC_CLEAR      (0x02 << 18)
#define ACPC_TOGGLE     (0x03 << 18)
#define ACPC            (0x03 << 18)

/* AEEVT : External event effect on TIOA */
#define AEEVT_NONE      (0x00 << 20)
#define AEEVT_SET       (0x01 << 20)
#define AEEVT_CLEAR     (0x02 << 20)
#define AEEVT_TOGGLE    (0x03 << 20)
#define AEEVT           (0x03 << 20)

/* ASWTRG : Software trigger effect on TIOA */
#define ASWTRG_NONE     (0x00 << 22)
#define ASWTRG_SET      (0x01 << 22)
#define ASWTRG_CLEAR    (0x02 << 22)
#define ASWTRG_TOGGLE   (0x03 << 22)
#define ASWTRG          (0x03 << 22)

/* BCPB : RB compare effect on TIOB */
#define BCPB_NONE       (0x00 << 24)
#define BCPB_SET        (0x01 << 24)
#define BCPB_CLEAR      (0x02 << 24)
#define BCPB_TOGGLE     (0x03 << 24)
#define BCPB            (0x03 << 24)

/* BCPC : RC compare effect on TIOB */
#define BCPC_NONE       (0x00 << 26)
#define BCPC_SET        (0x01 << 26)
#define BCPC_CLEAR      (0x02 << 26)
#define BCPC_TOGGLE     (0x03 << 26)
#define BCPC            (0x03 << 26)

/* BEEVT : External event effect on TIOB */
#define BEEVT_NONE      (0x00 << 28)
#define BEEVT_SET       (0x01 << 28)
#define BEEVT_CLEAR     (0x02 << 28)
#define BEEVT_TOGGLE    (0x03 << 28)
#define BEEVT           (0x03 << 28)

/* ASWTRG : Software trigger effect on TIOA */
#define BSWTRG_NONE     (0x00 << 30)
#define BSWTRG_SET      (0x40000000)
#define BSWTRG_CLEAR    (0x80000000)
#define BSWTRG_TOGGLE   (0xC0000000)
#define BSWTRG          (0xC0000000)


/******************************************************************************
* Timer Channel Status Register (capture & waveform mode)
******************************************************************************/
#define COVFS           (0x01 << 0)   /* Counter overflow                    */
#define LOVRS           (0x01 << 1)   /* Load overrun                        */
#define CPAS            (0x01 << 2)   /* RA compare                          */
#define CPBS            (0x01 << 3)   /* RB compare                          */
#define CPCS            (0x01 << 4)   /* RC compare                          */
#define LDRAS           (0x01 << 5)   /* RA loading                          */
#define LDRBS           (0x01 << 6)   /* RB loading                          */
#define ETRGS           (0x01 << 7)   /* External trigger                    */
#define CLKSTA          (0x01 << 8)   /* CLOCK enabling                      */
#define MTIOA           (0x01 << 9)   /* TIOA mirror                         */
#define MTIOB           (0x01 << 10)  /* TIOB mirror                         */


/******************************************************************************
* Timer Channel Counter Register  CV
******************************************************************************/
#define COUNTER_VAL     (0xFFFF << 0) /* Counter value mask                  */
      
                                                                             
/******************************************************************************
* Timer Control Register in test mode TSTC
******************************************************************************/
#define LDCT            (0x01 << 0)   /* Load timer counter (T1C)            */
#define LDCT0           (0x01 << 0)   /* Load timer 0 counter (T3C)          */
#define LDCT1           (0x01 << 1)   /* Load timer 1 counter (T3C)          */
#define LDCT2           (0x01 << 2)   /* Load timer 2 counter (T3C)          */


/******************************************************************************
* Timer Mode Register in test mode TSTM
******************************************************************************/
#define OCLKEN          (0x01 << 0)   /* Output clock on TIOB0 (T1C)         */
#define OCLKEN0         (0x01 << 0)   /* Output clock on TIOB0 for timer 0 (T3C) */
#define OCLKEN1         (0x01 << 1)   /* Output clock on TIOB0 for timer 1 (T3C) */
#define OCLKEN2         (0x01 << 2)   /* Output clock on TIOB0 for timer 2 (T3C) */


/******************************************************************************
* Timer Test Block Control Register BCR
******************************************************************************/
/* SYNC  : Synchro Command */
#define TC_SWRST        (0x01 << 0)   /* TC software reset                   */
#define TCSYNC          (0x01 << 1)   /* TC synchronization                  */


/******************************************************************************
* Timer Test Block Mode Register  BMR
******************************************************************************/
/* TC0XC0S  : External clock signal 0 selection */
#define TCLK0_XC0       (0x00 << 0)
#define NONE_XC0        (0x01 << 0)
#define TIOA1_XC0       (0x02 << 0)
#define TIOA2_XC0       (0x03 << 0)
#define TC0XC0S         (0x03 << 0)

/* TC1XC1S  : External clock signal 1 selection */
#define TCLK1_XC1       (0x00 << 2)
#define NONE_XC1        (0x01 << 2)
#define TIOA0_XC1       (0x02 << 2)
#define TIOA2_XC1       (0x03 << 2)
#define TC1XC1S         (0x03 << 2)

/* TC2XC2S  : External clock signal 2 selection */
#define TCLK2_XC2       (0x00 << 4)
#define NONE_XC2        (0x01 << 4)
#define TIOA0_XC2       (0x02 << 4)
#define TIOA1_XC2       (0x03 << 4)
#define TC2XC2S         (0x03 << 4)



/******************************************************************************
* Define LED
******************************************************************************/
#define RED     TIOA                  /* Red    LED correspond to TIOA pin   */
#define ORANGE  TIOB                  /* Orange LED correspond to TIOB pin   */
#define GREEN   TIOA                  /* Green  LED correspond to TIOA pin   */


#define RTOTC0_BASE_ADDRESS           (CSP_RTOT3C_T *)0xFFFC8000  /* 16-bit timer T0 channel 0   */

#define N0CHANNEL  0
#define N1CHANNEL  1

#define CPU_FREQ 32768000
#define CLOCK_FREQ  100 // 10 ticks per sec