/* atof_vax.c - */

/* Copyright (C) 1987 Free Software Foundation, Inc.

This file is part of Gas, the GNU Assembler.

The GNU assembler is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU Assembler General
Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the GNU Assembler, but only under the conditions described in the
GNU Assembler General Public License.  A copy of this license is
supposed to have been given to you along with the GNU Assembler
so you can know your rights and responsibilities.  It should be
in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.  */

	/* JF added these two for md_atof() */
#include "as.h"
#include "read.h"

#include "flonum.h"


				/* Precision in LittleNums. */
#define MAX_PRECISION (8)
#define H_PRECISION (8)
#define G_PRECISION (4)
#define D_PRECISION (4)
#define F_PRECISION (2)

				/* Length in LittleNums of guard bits. */
#define GUARD (2)

int				/* Number of chars in flonum type 'letter'. */
atof_vax_sizeof (letter)
     char letter;
{
  int	return_value;

  /*
   * Permitting uppercase letters is probably a bad idea.
   * Please use only lower-cased letters in case the upper-cased
   * ones become unsupported!
   */
  switch (letter)
    {
    case 'f':
    case 'F':
      return_value = 4;
      break;

    case 'd':
    case 'D':
    case 'g':
    case 'G':
      return_value = 8;
      break;

    case 'h':
    case 'H':
      return_value = 16;
      break;

    default:
      return_value = 0;
      break;
    }
  return (return_value);
}				/* atof_vax_sizeof */

static long int mask [] = {
  0x00000000,
  0x00000001,
  0x00000003,
  0x00000007,
  0x0000000f,
  0x0000001f,
  0x0000003f,
  0x0000007f,
  0x000000ff,
  0x000001ff,
  0x000003ff,
  0x000007ff,
  0x00000fff,
  0x00001fff,
  0x00003fff,
  0x00007fff,
  0x0000ffff,
  0x0001ffff,
  0x0003ffff,
  0x0007ffff,
  0x000fffff,
  0x001fffff,
  0x003fffff,
  0x007fffff,
  0x00ffffff,
  0x01ffffff,
  0x03ffffff,
  0x07ffffff,
  0x0fffffff,
  0x1fffffff,
  0x3fffffff,
  0x7fffffff,
  0xffffffff
  };

static int
next_bits (number_of_bits, address_of_bits_left_in_littlenum, address_of_littlenum_pointer)
     int		number_of_bits;
     int *		address_of_bits_left_in_littlenum;
     LITTLENUM_TYPE **	address_of_littlenum_pointer;
{
  int			return_value;

  if (number_of_bits >= (* address_of_bits_left_in_littlenum))
    {
      return_value  = mask [(* address_of_bits_left_in_littlenum)] & * (* address_of_littlenum_pointer);
      number_of_bits -= (* address_of_bits_left_in_littlenum);
      return_value <<= number_of_bits;
      (* address_of_bits_left_in_littlenum) = LITTLENUM_NUMBER_OF_BITS - number_of_bits;
      (* address_of_littlenum_pointer) --;
      return_value |= ( (* (* address_of_littlenum_pointer)) >> ((* address_of_bits_left_in_littlenum)) ) & mask [number_of_bits];
    }
  else
    {
      (* address_of_bits_left_in_littlenum) -= number_of_bits;
      return_value = mask [number_of_bits] & ( (* (* address_of_littlenum_pointer)) >> (* address_of_bits_left_in_littlenum));
    }
  return (return_value);
}

static void
make_invalid_floating_point_number (words)
     LITTLENUM_TYPE *	words;
{
  * words = 0x8000;		/* Floating Reserved Operand Code */
}

static int			/* 0 means letter is OK. */
what_kind_of_float (letter, precisionP, exponent_bitsP)
     char		letter;	/* In: lowercase please. What kind of float? */
     int *		precisionP; /* Number of 16-bit words in the float. */
     long int *		exponent_bitsP;	/* Number of exponent bits. */
{
  int	retval;			/* 0: OK. */

  retval = 0;
  switch (letter)
    {
    case 'f':
      * precisionP = F_PRECISION;
      * exponent_bitsP = 8;
      break;

    case 'd':
      * precisionP = D_PRECISION;
      * exponent_bitsP = 8;
      break;

    case 'g':
      * precisionP = G_PRECISION;
      * exponent_bitsP = 11;
      break;

    case 'h':
      * precisionP = H_PRECISION;
      * exponent_bitsP = 15;
      break;

    default:
      retval = 69;
      break;
    }
  return (retval);
}

/***********************************************************************\
*									*
*	Warning: this returns 16-bit LITTLENUMs, because that is	*
*	what the VAX thinks in. It is up to the caller to figure	*
*	out any alignment problems and to conspire for the bytes/word	*
*	to be emitted in the right order. Bigendians beware!		*
*									*
\***********************************************************************/

char *				/* Return pointer past text consumed. */
atof_vax (str, what_kind, words)
     char *		str;	/* Text to convert to binary. */
     char		what_kind; /* 'd', 'f', 'g', 'h' */
     LITTLENUM_TYPE *	words;	/* Build the binary here. */
{
  FLONUM_TYPE		f;
  LITTLENUM_TYPE	bits [MAX_PRECISION + MAX_PRECISION + GUARD];
				/* Extra bits for zeroed low-order bits. */
				/* The 1st MAX_PRECISION are zeroed, */
				/* the last contain flonum bits. */
  char *		return_value;
  int			precision; /* Number of 16-bit words in the format. */
  long int		exponent_bits;

  return_value = str;
  f . low	= bits + MAX_PRECISION;
  f . high	= NULL;
  f . leader	= NULL;
  f . exponent	= NULL;
  f . sign	= '\0';

  if (what_kind_of_float (what_kind, & precision, & exponent_bits))
    {
      return_value = NULL;	/* We lost. */
      make_invalid_floating_point_number (words);
    }
  if (return_value)
    {
      bzero (bits, sizeof(LITTLENUM_TYPE) * MAX_PRECISION);

				/* Use more LittleNums than seems */
				/* necessary: the highest flonum may have */
				/* 15 leading 0 bits, so could be useless. */
      f . high = f . low + precision - 1 + GUARD;

      if (atof_generic (& return_value, ".", "eE", & f))
	{
	  make_invalid_floating_point_number (words);
	  return_value = NULL;	/* we lost */
	}
      else
	{
	  if (flonum_gen2vax (what_kind, & f, words))
	    {
	      return_value = NULL;
	    }
	}
    }
  return (return_value);
}

/*
 * In: a flonum, a vax floating point format.
 * Out: a vax floating-point bit pattern.
 */

int				/* 0: OK. */
flonum_gen2vax (format_letter, f, words)
     char		format_letter; /* One of 'd' 'f' 'g' 'h'. */
     FLONUM_TYPE *	f;
     LITTLENUM_TYPE *	words;	/* Deliver answer here. */
{
  int			bits_left_in_littlenum;
  LITTLENUM_TYPE *	littlenum_pointer;
  LITTLENUM_TYPE *	lp;
  int			precision;
  long int		exponent_bits;
  int			return_value; /* 0 == OK. */

  return_value = what_kind_of_float (format_letter, & precision, & exponent_bits);
  if (return_value != 0)
    {
      make_invalid_floating_point_number (words);
    }
  else
    {
      if (f -> low > f -> leader)
	{
	  /* 0.0e0 seen. */
	  bzero (words, sizeof(LITTLENUM_TYPE) * precision);
	}
      else
	{
	  long int		exponent_1;
	  long int		exponent_2;
	  long int		exponent_3;
	  long int		exponent_4;
	  int		exponent_skippage;
	  LITTLENUM_TYPE	word1;
	  
	  /*
	   * All vaxen floating_point formats (so far) have:
	   * Bit 15 is sign bit.
	   * Bits 14:n are excess-whatever exponent.
	   * Bits n-1:0 (if any) are most significant bits of fraction.
	   * Bits 15:0 of the next word are the next most significant bits.
	   * And so on for each other word.
	   *
	   * All this to be compatible with a KF11?? (Which is still faster
	   * than lots of vaxen I can think of, but it also has higher
	   * maintenance costs ... sigh).
	   *
	   * So we need: number of bits of exponent, number of bits of
	   * mantissa.
	   */
	  
#ifdef NEVER  /******* This zeroing seems redundant - Dean 3may86 **********/
	  /*
	   * No matter how few bits we got back from the atof()
	   * routine, add enough zero littlenums so the rest of the
	   * code won't run out of "significant" bits in the mantissa.
	   */
	  {
	    LITTLENUM_TYPE * ltp;
	    for (ltp = f -> leader + 1;
		 ltp <= f -> low + precision;
		 ltp ++)
	      {
		* ltp = 0;
	      }
	  }
#endif
	  
	  bits_left_in_littlenum = LITTLENUM_NUMBER_OF_BITS;
	  littlenum_pointer = f -> leader;
	  /* Seek (and forget) 1st significant bit */
	  for (exponent_skippage = 0;
	       ! next_bits(1, &bits_left_in_littlenum, &littlenum_pointer);
	       exponent_skippage ++)
	    {
	    }
	  exponent_1 = f -> exponent + f -> leader + 1 - f -> low;
	  /* Radix LITTLENUM_RADIX, point just higher than f -> leader. */
	  exponent_2 = exponent_1 * LITTLENUM_NUMBER_OF_BITS;
	  /* Radix 2. */
	  exponent_3 = exponent_2 - exponent_skippage;
	  /* Forget leading zeros, forget 1st bit. */
	  exponent_4 = exponent_3 + (1 << (exponent_bits - 1));
	  /* Offset exponent. */
	  
	  if (exponent_4 & ~ mask [exponent_bits])
	    {
	      /*
	       * Exponent overflow. Lose immediately.
	       */
	      
	      make_invalid_floating_point_number (words);
	      
	      /*
	       * We leave return_value alone: admit we read the
	       * number, but return a floating exception
	       * because we can't encode the number.
	       */
	    }
	  else
	    {
	      lp = words;
	      
	      /* Word 1. Sign, exponent and perhaps high bits. */
	      /* Assume 2's complement integers. */
	      word1 = ((exponent_4 & mask [exponent_bits]) << (15 - exponent_bits))
		|       ((f -> sign == '+') ? 0 : 0x8000)
		  |	next_bits (15 - exponent_bits, &bits_left_in_littlenum, &littlenum_pointer);
	      * lp ++ = word1;
	      
	      /* The rest of the words are just mantissa bits. */
	      for (; lp < words + precision; lp++)
		{
		  * lp = next_bits (LITTLENUM_NUMBER_OF_BITS, &bits_left_in_littlenum, &littlenum_pointer);
		}
	      
	      if (next_bits (1, &bits_left_in_littlenum, &littlenum_pointer))
		{
		  /*
		   * Since the NEXT bit is a 1, round UP the mantissa.
		   * The cunning design of these hidden-1 floats permits
		   * us to let the mantissa overflow into the exponent, and
		   * it 'does the right thing'. However, we lose if the
		   * highest-order bit of the lowest-order word flips.
		   * Is that clear?
		   */
		  
		  unsigned long int	carry;
		  
		  /*
		    #if (sizeof(carry)) < ((sizeof(bits[0]) * BITS_PER_CHAR) + 2)
		    Please allow at least 1 more bit in carry than is in a LITTLENUM.
		    We need that extra bit to hold a carry during a LITTLENUM carry
		    propagation. Another extra bit (kept 0) will assure us that we
		    don't get a sticky sign bit after shifting right, and that
		    permits us to propagate the carry without any masking of bits.
		    #endif
		    */
		  for (carry = 1, lp --;
		       carry && (lp >= words);
		       lp --)
		    {
		      carry = * lp + carry;
		      * lp = carry;
		      carry >>= LITTLENUM_NUMBER_OF_BITS;
		    }
		  
		  if ( (word1 ^ *words) & (1 << (LITTLENUM_NUMBER_OF_BITS - 1)) )
		    {
		      make_invalid_floating_point_number (words);
		      /*
		       * We leave return_value alone: admit we read the
		       * number, but return a floating exception
		       * because we can't encode the number.
		       */
		    }
		}		/* if (we needed to round up) */
	    }			/* if (exponent overflow) */
	}			/* if (0.0e0) */
    }				/* if (float_type was OK) */
  return (return_value);
}


/* JF this used to be in vax.c but this looks like a better place for it */

/*
 *		md_atof()
 *
 * In:	input_line_pointer -> the 1st character of a floating-point
 *		number.
 *	1 letter denoting the type of statement that wants a
 *		binary floating point number returned.
 *	Address of where to build floating point literal.
 *		Assumed to be 'big enough'.
 *	Address of where to return size of literal (in chars).
 *
 * Out:	Input_line_pointer -> of next char after floating number.
 *	Error message, or "".
 *	Floating point literal.
 *	Number of chars we used for the literal.
 */

int atof_vax_sizeof();

#define MAXIMUM_NUMBER_OF_LITTLENUMS (8) /* For .hfloats. */

char *
md_atof (what_statement_type, literalP, sizeP)
     char	what_statement_type;
     char *	literalP;
     int *	sizeP;
{
  LITTLENUM_TYPE	words [MAXIMUM_NUMBER_OF_LITTLENUMS];
  register char		kind_of_float;
  register int		number_of_chars;
  register LITTLENUM_TYPE * littlenum_pointer;

  switch (what_statement_type)
    {
    case 'F':			/* .float */
    case 'f':			/* .ffloat */
      kind_of_float = 'f';
      break;

    case 'D':			/* .double */
    case 'd':			/* .dfloat */
      kind_of_float = 'd';
      break;

    case 'g':			/* .gfloat */
      kind_of_float = 'g';
      break;

    case 'h':			/* .hfloat */
      kind_of_float = 'h';
      break;

    default:
      kind_of_float = 0;
      break;
    };

  if (kind_of_float)
    {
      register LITTLENUM_TYPE * limit;
      char * atof_vax();

      input_line_pointer = atof_vax (input_line_pointer,
				     kind_of_float,
				     words);
      /*
       * The atof_vax() builds up 16-bit numbers.
       * Since the assembler may not be running on
       * a little-endian machine, be very careful about
       * converting words to chars.
       */
      number_of_chars = atof_vax_sizeof (kind_of_float);
      know( number_of_chars <= MAXIMUM_NUMBER_OF_LITTLENUMS * sizeof(LITTLENUM_TYPE) );
      limit = words + (number_of_chars / sizeof(LITTLENUM_TYPE));
      for (littlenum_pointer = words;
	   littlenum_pointer < limit;
	   littlenum_pointer ++)
	{
	  md_number_to_chars (literalP, * littlenum_pointer, sizeof(LITTLENUM_TYPE));
	  literalP += sizeof(LITTLENUM_TYPE);
	};
    }
  else
    {
      number_of_chars = 0;
    };

  * sizeP = number_of_chars;
  return (kind_of_float ? "" : "Bad call to md_atof()");
}				/* md_atof() */

/* atof_vax.c */
