/* ldmisc.c
   Copyright (C) 1991, 92, 93, 94 Free Software Foundation, Inc.

   Written by Steve Chamberlain of Cygnus Support.

This file is part of GLD, the Gnu Linker.

GLD is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GLD is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GLD; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "bfd.h"
#include "sysdep.h"
#include <varargs.h>
#include <demangle.h>

#include "ld.h"
#include "ldmisc.h"
#include "ldexp.h"
#include "ldlang.h"
#include "ldgram.h"
#include "ldlex.h"
#include "ldmain.h"
#include "ldfile.h"

/* VARARGS*/
static void finfo ();
static const char *demangle PARAMS ((const char *string,
				     int remove_underscore));

/*
 %% literal %
 %F error is fatal
 %P print program name
 %S print script file and linenumber
 %E current bfd error or errno
 %I filename from a lang_input_statement_type
 %B filename from a bfd
 %T symbol name
 %X no object output, fail return
 %V hex bfd_vma
 %v hex bfd_vma, no leading zeros
 %C Clever filename:linenumber 
 %R info about a relent
 %s arbitrary string, like printf
 %d integer, like printf
*/

static const char *
demangle (string, remove_underscore)
     const char *string;
     int remove_underscore;
{
  const char *res;
  if (remove_underscore && output_bfd) 
  {
    if (bfd_get_symbol_leading_char (output_bfd) == string[0])
     string++;
  }
  /* Note that there's a memory leak here, we keep buying memory
     for demangled names, and never free.  But if you have so many
     errors that you run out of VM with the error messages, then
     there's something up */
  res = cplus_demangle (string, DMGL_ANSI|DMGL_PARAMS);
  return res ? res : string;
}

static void
vfinfo(fp, fmt, arg)
     FILE *fp;
     char *fmt;
     va_list arg;
{
  boolean fatal = false;

  while (*fmt) 
  {
    while (*fmt != '%' && *fmt != '\0') 
    {
      putc(*fmt, fp);
      fmt++;
    }

    if (*fmt == '%') 
    {
      fmt ++;
      switch (*fmt++) 
      {
      default:
	fprintf(fp,"%%%c", fmt[-1]);
	break;

      case '%':
	/* literal % */
	putc('%', fp);
	break;

       case 'X':
	/* no object output, fail return */
	config.make_executable = false;
	break;

       case 'V':
	/* hex bfd_vma */
	{
	  bfd_vma value = va_arg(arg, bfd_vma);
	  fprintf_vma(fp, value);
	}
	break;

      case 'v':
	/* hex bfd_vma, no leading zeros */
	{
	  char buf[100];
	  char *p = buf;
	  bfd_vma value = va_arg (arg, bfd_vma);
	  sprintf_vma (p, value);
	  while (*p == '0')
	    p++;
	  if (!*p)
	    p--;
	  fputs (p, fp);
	}
	break;

       case 'T':
	/* Symbol name.  */
	{
	  const char *name = va_arg (arg, const char *);

	  if (name != (const char *) NULL)
	    fprintf (fp, "%s", demangle (name, 1));
	  else
	    fprintf (fp, "no symbol");
	}
	break;

       case 'B':
	/* filename from a bfd */
       { 
	 bfd *abfd = va_arg(arg, bfd *);
	 if (abfd->my_archive) {
	   fprintf(fp,"%s(%s)", abfd->my_archive->filename,
		   abfd->filename);
	 }
	 else {
	   fprintf(fp,"%s", abfd->filename);
	 }
       }
	break;

       case 'F':
	/* error is fatal */
	fatal = true;
	break;

       case 'P':
	/* print program name */
	fprintf(fp,"%s", program_name);
	break;

       case 'E':
	/* current bfd error or errno */
	fprintf(fp, bfd_errmsg(bfd_get_error ()));
	break;

       case 'I':
	/* filename from a lang_input_statement_type */
       {
	 lang_input_statement_type *i =
	  va_arg(arg,lang_input_statement_type *);

	 if (i->the_bfd->my_archive)
	   fprintf(fp, "(%s)", i->the_bfd->my_archive->filename);
	 fprintf(fp,"%s", i->local_sym_name);
       }
	break;

       case 'S':
	/* print script file and linenumber */
       {
	 if (ldfile_input_filename) {
	   fprintf(fp,"%s:%u", ldfile_input_filename, lineno );
	 }
       }
	break;

       case 'R':
	/* Print all that's interesting about a relent */
       {
	 arelent *relent = va_arg(arg, arelent *);
	
	 finfo (fp, "%s+0x%v (type %s)",
		(*(relent->sym_ptr_ptr))->name,
		relent->addend,
		relent->howto->name);
       }
	break;
	
       case 'C':
	/* Clever filename:linenumber with function name if possible,
	   or section name as a last resort.  The arguments are a BFD,
	   a section, and an offset.  */
	{
	  bfd *abfd;
	  asection *section;
	  bfd_vma offset;
	  lang_input_statement_type *entry;
	  asymbol **asymbols;
	  const char *filename;
	  const char *functionname;
	  unsigned int linenumber;

	  abfd = va_arg (arg, bfd *);
	  section = va_arg (arg, asection *);
	  offset = va_arg (arg, bfd_vma);

	  entry = (lang_input_statement_type *) abfd->usrdata;
	  if (entry != (lang_input_statement_type *) NULL
	      && entry->asymbols != (asymbol **) NULL)
	    asymbols = entry->asymbols;
	  else
	    {
	      long symsize;
	      long symbol_count;

	      symsize = bfd_get_symtab_upper_bound (abfd);
	      if (symsize < 0)
		einfo ("%B%F: could not read symbols", abfd);
	      asymbols = (asymbol **) xmalloc (symsize);
	      symbol_count = bfd_canonicalize_symtab (abfd, asymbols);
	      if (symbol_count < 0)
		einfo ("%B%F: could not read symbols", abfd);
	      if (entry != (lang_input_statement_type *) NULL)
		{
		  entry->asymbols = asymbols;
		  entry->symbol_count = symbol_count;
		}
	    }

	  if (bfd_find_nearest_line (abfd, section, asymbols, offset,
				     &filename, &functionname, &linenumber))
	    {
	      if (filename == (char *) NULL)
		filename = abfd->filename;

	      if (functionname != (char *)NULL) 
		fprintf (fp, "%s:%u: %s", filename, linenumber,
			 demangle (functionname, 1));
	      else if (linenumber != 0) 
		fprintf (fp, "%s:%u", filename, linenumber);
	      else
		finfo (fp, "%s(%s+0x%v)", filename, section->name, offset);

	    }
	  else
	    finfo (fp, "%s(%s+0x%v)", abfd->filename, section->name, offset);
	}
	break;
		
       case 's':
	/* arbitrary string, like printf */
	fprintf(fp,"%s", va_arg(arg, char *));
	break;

       case 'd':
	/* integer, like printf */
	fprintf(fp,"%d", va_arg(arg, int));
	break;
      }
    }
  }

  if (fatal == true) 
    xexit(1);
}

/* Format info message and print on stdout. */

/* (You would think this should be called just "info", but then you would
   hosed by LynxOS, which defines that name in its libc.) */

void info_msg(va_alist)
     va_dcl
{
  char *fmt;
  va_list arg;
  va_start(arg);
  fmt = va_arg(arg, char *);
  vfinfo(stdout, fmt, arg);
  va_end(arg);
}

/* ('e' for error.) Format info message and print on stderr. */

void einfo(va_alist)
     va_dcl
{
  char *fmt;
  va_list arg;
  va_start(arg);
  fmt = va_arg(arg, char *);
  vfinfo(stderr, fmt, arg);
  va_end(arg);
}

/* Warn about a symbol NEWSYM being multiply defined with another symbol OLDSYM.
   MESSAGE1 and MESSAGE2 should look something like:
   "%C: warning: multiple commons of `%s'\n"
   "%C: warning: previous common here\n"  */

void
multiple_warn (message1, newsym, message2, oldsym)
     char *message1;
     asymbol *newsym;
     char *message2;
     asymbol *oldsym;
{
  lang_input_statement_type *inp_stat;
  asymbol **stat_symbols;

  inp_stat = (lang_input_statement_type *) bfd_asymbol_bfd (newsym)->usrdata;
  stat_symbols = inp_stat ? inp_stat->asymbols : 0;

  einfo (message1,
	 bfd_asymbol_bfd (newsym), newsym->section, stat_symbols, newsym->value,
	 demangle (newsym->name, 1));

  inp_stat = (lang_input_statement_type *) bfd_asymbol_bfd (oldsym)->usrdata;
  stat_symbols = inp_stat ? inp_stat->asymbols : 0;

  einfo (message2,
	 bfd_asymbol_bfd (oldsym), oldsym->section, stat_symbols, oldsym->value);
}

void 
info_assert(file, line)
     char *file;
     unsigned int line;
{
  einfo("%F%P: internal error %s %d\n", file,line);
}

char *
buystring (x)
     CONST char *CONST x;
{
  size_t l = strlen(x)+1;
  char *r = xmalloc(l);
  memcpy(r, x,l);
  return r;
}


/* ('m' for map) Format info message and print on map. */

void minfo(va_alist)
     va_dcl
{
  char *fmt;
  va_list arg;
  va_start(arg);
  fmt = va_arg(arg, char *);
  vfinfo(config.map_file, fmt, arg);
  va_end(arg);
}


static void
finfo (va_alist)
     va_dcl
{
  char *fmt;
  FILE *file;
  va_list arg;
  va_start (arg);
  file = va_arg (arg, FILE *);
  fmt = va_arg (arg, char *);
  vfinfo (file, fmt, arg);
  va_end (arg);
}



/*----------------------------------------------------------------------
  Functions to print the link map 
 */

void 
print_space ()
{
  fprintf(config.map_file, " ");
}
void 
print_nl ()
{
  fprintf(config.map_file, "\n");
}
void 
print_address (value)
     bfd_vma value;
{
  fprintf_vma(config.map_file, value);
}
