/*
 * ARCload (c) 2005 Stanislaw Skowronek
 */

#include <arclib/arc.h>
#include <arclib/stddef.h>
#include <arclib/stdio.h>
#include <arclib/stdlib.h>
#include <arclib/types.h>
#include <arclib/string.h>
#include <elf.h>

#include "block.h"
#include "config.h"
#include "detect.h"

#define VERSION		"0.5"

#define N_LABEL		16
#define N_LABEL_SPACE	256
#define N_IMAGE		256
#define N_APPEND	32
#define N_APPEND_SPACE	2048

#define PAGESHIFT	12
#define READBLOCK	16384

#define N_PHDR		16

// malloc pool size
#define ARENA		1048576

INT checkmemmap(ULONG elf_start, ULONG elf_size)
{
	MEMORYDESCRIPTOR *md;
	ULONG md_start = 0, md_size = 0, md_end = 0;
	ULONG elf_end = (elf_start + elf_size);

	md = ArcGetMemoryDescriptor(NULL);
	while (md) {
		md_start	= (md->BasePage << PAGESHIFT);
		md_size		= (md->PageCount << PAGESHIFT);
		md_end		= (md_start + md_size);

		if((elf_start >= md_start) && (elf_start <= md_end)) {
			if((md->Type != FreeContiguous) && (md->Type != FreeMemory)) {
				printf("Memory not free at given address.\n\r");
				return -1;
			}
			if(elf_end > md_end) {
				printf("Free memory area at given address is too small.\n\r");
				return -1;
			}
			return 0;
		}
		md = ArcGetMemoryDescriptor(md);
	}
	printf("No descriptor for this range found.\n\r");
	return -1;
}

ULONG vtophys(ULONG addr)
{
#ifdef M32
	if((addr & 0xC0000000) != 0x80000000)
		printf("Warning: kernel is in a mapped area!\n\r");

	return (addr & 0x1FFFFFFF);
#else
	/* xkphys */
	if((addr >> 62) == 2)
		return (addr & ((1UL << 40) - 1));

	/* ckseg */
	if((addr >> 60) == 15) {
		if((addr & 0xC0000000UL) != 0x80000000UL)
			printf("Warning: kernel is in a mapped area!\n\r");

		return (addr & 0x1FFFFFFF);
	}

	printf("Warning: kernel is in a mapped area!\n\r");
	return (addr & ((1UL << 40) - 1));
#endif
}

ULONG load_elf(CHAR *fname)
{
	BDEV file;
#ifdef M32
	Elf32_Ehdr hdr;
	Elf32_Phdr ph[N_PHDR];
	ULONG elf_ehdr_size = sizeof(Elf32_Ehdr);
	ULONG elf_phdr_size = sizeof(Elf32_Phdr);
#else
	Elf64_Ehdr hdr;
	Elf64_Phdr ph[N_PHDR];
	ULONG elf_ehdr_size = sizeof(Elf64_Ehdr);
	ULONG elf_phdr_size = sizeof(Elf64_Phdr);
#endif
	INT i, j, nph = 0;
	ULONG left, pos;
	UCHAR *ptr;

	/* Open */
	if(bopen(&file, fname)) {
		printf("Opening %s failed.\n\r", fname);
		return 0;
	}

	printf("Loading %s...\n\r", fname);

	/* Read ELF Header */
	if(bread(&file, elf_ehdr_size, &hdr)<0) {
		printf("Error reading ELF header.\n\r");
		goto close;
	}

	/* Is is an ELF binary? */
	if((hdr.e_ident[0] != 0x7f) || 
	   (hdr.e_ident[1] != 'E')  || 
	   (hdr.e_ident[2] != 'L')  ||
	   (hdr.e_ident[3] != 'F'))
	{
		printf("ELF magic invalid.\n\r");
		goto close;
	}

	/* 32 or 64 bit? */
#ifdef M32
	if(hdr.e_ident[EI_CLASS] != ELFCLASS32) {
		printf("ELF file is not 32-bit.\n\r");
#else
	if(hdr.e_ident[EI_CLASS] != ELFCLASS64) {
		printf("ELF file is not 64-bit.\n\r");
#endif
		goto close;
	}

	/* Big Endian? */
	if(hdr.e_ident[EI_DATA] != ELFDATA2MSB) {
		printf("ELF file is not big-endian.\n\r");
		goto close;
	}

	/* Not Relocatable? */
	if(hdr.e_ident[EI_DATA] != ET_EXEC) {
		printf("ELF file is not executable.\n\r");
		printf("Relocatable files are not supported, sorry.\n\r");
		goto close;
	}

	/* Does the ELF have any Program Headers? */
	if(hdr.e_phnum == 0) {
		printf("ELF file contains no Program Headers.\n\r");
		goto close;
	}

	/* Save LOAD headers */
	for(i = 0; i < hdr.e_phnum; i++) {
		bseek(&file, (hdr.e_phoff + (i * hdr.e_phentsize)));
		if(bread(&file, elf_phdr_size, ph+nph)<0) {
			printf("Can't read program header %u at 0x%x!\n\r", i, hdr.e_phoff + (i * hdr.e_phentsize));
			goto close;
		}

		if(ph[nph].p_type != PT_LOAD)
			continue;

		nph++;
		if(nph>=N_PHDR) {
			printf("ELF file has too many LOAD headers (over N_PHDR).\n\r");
			goto close;
		}
	}

	/* Was a LOAD header found? */
	if(!nph) {
		printf("ELF file contains no LOAD header.\n\r");
		goto close;
	}

	/* Realize LOAD headers */
	for(i = 0; i < nph; i++) {
		/* Check the Memory Map */
		if(checkmemmap(vtophys(ph[i].p_vaddr), ph[i].p_memsz)) {
			printf("File can't be loaded into current memory map.\n\r");
			goto close;
		}

		/* Load the ELF into memory */
		printf("Reading %u bytes... ", (ULONG)ph[i].p_filesz);
		left = ph[i].p_filesz;
		pos = 0;
		ptr = (UCHAR *)ph[i].p_vaddr;
		bseek(&file, ph[i].p_offset);
		while(left > 0) {
			j = ((left > READBLOCK) ? READBLOCK : left);
			if(bread(&file, j, ptr)<0) {
				printf("failed at %u!\n\r", (ULONG)pos);
				goto close;
			}
			left -= j;
			ptr += j;
			pos += j;
		}
		printf("OK.\n\r");
		ptr = (UCHAR *)(ph[i].p_vaddr + ph[i].p_filesz);
		for(pos = ph[i].p_filesz; pos < ph[i].p_memsz; pos++)
			*(ptr++) = 0;
	}

	bclose(&file);

	return hdr.e_entry;

/* Failed to do something, close and return */
close:
	bclose(&file);
	return 0;
}

#ifdef M64
unsigned fixup_trampolines_data[]={
	0x03e00821, 0x04110001, 0x00000000, 0xdfe30014,
	0xdfe2001c, 0x0060c821, 0x00600008, 0x0020f821};

extern void *__rodata_start;
extern void *__rodata_end;
static void fixup_trampolines(void)
{
	unsigned *start=(unsigned *)&__rodata_start;
	unsigned *end=(unsigned *)&__rodata_end;
	int stat=0;
	while(start<end) {
		if(!memcmp(start,fixup_trampolines_data,32)) {
			start[0]=0x03e0082d;
			start[5]=0x0060c82d;
			start[7]=0x0020f82d;
			stat++;
		}
		start++;
	}
	if(stat)
		printf("-- WARNING --\n"
		       "Your version of GCC does not support nested functions.\n"
		       "Please apply patch or download new version.\n"
		       "Patched %u occurrences of the wrong trampoline.\n"
		       "-- WARNING --\n", stat);
}
#endif

void __start(void);
LONG main(INT argc, CHAR **argv, CHAR **envp)
{
	ULONG entry;
	VOID (*kernel_entry)(INT argc, CHAR **argv, CHAR **envp);

	char *system;

	char text_label[N_LABEL_SPACE];
	char *label[N_LABEL];
	char label_space[N_LABEL_SPACE];
	int label_count = 0;

	char image[N_IMAGE];
	char *append[N_APPEND];
	char append_space[N_APPEND_SPACE];
	int append_count;

	unsigned long arena_start;

	printf("ARCLoad version " VERSION " (c) 2004-5 Stanislaw Skowronek\n\r");

#ifdef M64
	fixup_trampolines();
#endif

	arena_start = (((ULONG)__start) - ARENA)&~((1UL<<PAGESHIFT)-1);
	if(checkmemmap(vtophys(arena_start), ARENA)) {
		printf("Internal error: dynamic memory arena allocation failed.\n\r");
		printf("Attempted: 0x%p bytes at 0x%p.\n\r", ARENA, vtophys(((ULONG)__start) - ARENA));
		return 1;
	}
	arclib_malloc_add(arena_start, ARENA);
	binitfs();

	if(!ArcGetEnvironmentVariable("tapedevice")) {
		if(!ArcGetEnvironmentVariable("OSLoadPartition")) {
			printf("Error: OSLoadPartition is not set.\n\r");
			return 1;
		}
		system = ArcGetEnvironmentVariable("OSLoadPartition");
	} else {
		system = ArcGetEnvironmentVariable("tapedevice");
	}
	if(ArcGetEnvironmentVariable("OSLoadFilename"))
		strcpy(text_label, ArcGetEnvironmentVariable("OSLoadFilename"));
	else
		strcpy(text_label, "no_system");

	if(argc>1 && !strcmp(argv[1],"list"))
		text_label[0] = 0;

	if(*text_label)
		printf("Loading configuration for '%s'...\n\r", text_label);
	parselabel(text_label, label, N_LABEL, &label_count, label_space, N_LABEL_SPACE);

	parseconfig(label, N_LABEL, &label_count, image, N_IMAGE, append, N_APPEND, append_space, N_APPEND_SPACE, &append_count, system);
	if(!*label)
		return 0;
	if(!*image) {
		printf("Failed.\n\r");
		return 1;
	}

	entry = load_elf(image);
	if(!entry)
		return 1;

	kernel_entry = (VOID(*)(INT, CHAR **, CHAR **))entry;

	printf("Entering kernel.\n\r");
	kernel_entry(append_count, append, envp);
	return 0;
}

/* don't ask */

void _flush_cache(void *addr, int n, int flag)
{
	ArcFlushAllCaches();
}
