# AMD Family 17h in coreboot

## Abstract

Beginning with Family 17h products (a.k.a. “Zen” cores), AMD
changed their paradigm for initializing the system and this requires
major modifications to the execution flow of coreboot.  This file
discusses the new boot flow, and challenges, and the tradeoffs of the
initial port into coreboot.

## Introduction

Family 17h products are x86-based designs.  This documentation assumes
familiarity with x86, its reset state and its early initialization
requirements.

To the extent necessary, the role of the AMD Secure Processor (a.k.a.
Platform Security Processor or PSP) in system initialization is addressed
here.  The PSP specification<sup>1</sup> is available only with an NDA.
coreboot relies on util/amdfwtool to build the structures and add various
other firmware to the final image<sup>2</sup>.  The Family 17h PSP design
guide adds a new BIOS Directory Table, similar to the PSP Directory Table.

Support in coreboot for modern AMD products is based on AMD’s
reference code:  AMD Generic Encapsulated Software Architecture
(AGESA<sup>TM</sup>).  AGESA contains the technology for enabling DRAM,
configuring proprietary core logic, assistance with generating ACPI
tables, and other features.

AGESA for products earlier than Family 17h is known as v5 or
Arch2008<sup>3</sup>.  Also note that coreboot currently contains both
open source AGESA and closed source implementations (binaryPI) compiled
from AGESA.

The first AMD Family 17h device ported to coreboot is codenamed
“Picasso”<sup>4</sup>, and will be added to soc/amd/picasso.

## Additional Definitions

*   PSP, Platform Security Processor:  Onboard ARM processor that runs
alongside the main x86 processor; may be viewed as analogous to the
Intel<sup>R</sup> Management Engine
*   FCH, Fusion Control Hub, the logical southbridge within the SOC
*   ABL - AGESA Bootloader - Processor initialization code that runs on
the PSP
*   PSP Directory Table - A structured list of pointers to PSP firmware
and other controller binaries
*   BIOS Directory Table - A structured list of pointers to BIOS
related firmware images
*   Embedded Firmware Structure - Signature and pointers used by the
PSP to locate the PSP Directory Table and BIOS Directory Table; these
items are generated during coreboot build and are located in the SPI ROM
*   vboot - The generic technology name for verifying/choosing a RW A/B
or fallback RO path.
*   verstage - The code (vboot) to verify the firmware contained in the
writable section of the SPI ROM, traditionally run on the x86 processor,
and in some cases a separate stage added to coreboot
*   vboot app - A portion of vboot technology designed and compiled
to run on the PSP
*   APCB - AMD PSP Customization Block - A binary containing PSP and
system configuration preferences (analogous to v5 BUILDOPT_ options),
and generated by APCBTool to be added to coreboot/utils later
*   APOB - AGESA PSP Output Buffer - A buffer in main memory for
storing AGESA BootLoader output.  There are no plans for this to be
parsed by coreboot

## Problem Statements

AMD has ported early AGESA features to the PSP, which now discovers,
enables and trains DRAM.  Unlike any other x86 device in coreboot, a
Picasso system has DRAM online prior to the first instruction fetch.

Cache-as-RAM (CAR) is no longer a supportable feature in AMD hardware.
Early code expecting CAR behavior <span
style="text-decoration:underline;">must</span> account for writes
escaping the L2 cache and going to DRAM.

Without any practical need for CAR, or DRAM initialization, coreboot
should arguably skip bootblock and romstage, and possibly use ramstage
as the BIOS image.  This approach presents a number of challenges:

*   At the entry of ramstage, x86 processors are in flat protected
mode.  Picasso’s initial state is nearly identical to any other x86
at reset, except its CS shadow register’s base and limit put its
execution within DRAM, not at 0xfffffff0.  Picasso requires initial
programming and entry into protected mode prior to ramstage.
*   coreboot expects cbmem initialization during romstage.

AGESA supporting Picasso is now at v9.  Unlike Arch2008, which defines
granular entry points for easy inclusion to a legacy BIOS, v9 is
rewritten for compilation into a UEFI.  The source follows UEFI
standards, i.e. assumes the presence of UEFI phases, implements
dependency expressions, much functionality is rewritten as libraries,
etc.  It would, in no way, fit into the v5 model used in coreboot.

*   For the foreseeable future, AGESA source will distributed only
under NDA.  Furthermore, because AGESA's integrated debug services divulge
NDA information, no debug builds will be released to the general public.

## Basic Pre-x86 Boot Flow

The following steps occur prior to x86 processor operation.

*   System power on
*   PSP executes immutable on-chip boot ROM
*   PSP locates the Embedded Firmware Table and PSP Directory Table in
the SPI ROM
*   PSP verifies and executes the PSP off-chip bootloader
*   ChromeOS systems:
    *   Off-chip bootloader attempts to locate vboot app via the RO BIOS
Directory Table
    *   If vboot app is not found, booting continues with ABLs below
    *   vboot app initializes, setting up GPIOs, UART if needed,
communication path to the EC, and the SPI controller for direct access
to the flash device.
    *   vboot app verifies the RW sections (as is typically performed by
the main processor)
    *   vboot app locates the Embedded Firmware Directory within the
verified FMAP section and passes a pointer to the PSP bootloader.  If
the verification fails, it passes a pointer to the RO header to the
bootloader.
*   PSP parses the PSP Directory Table to find the ABLs and executes
them
*   An ABL parses the APCB for system configuration preferences
*   An ABL initializes system main memory, locates the compressed BIOS
image in the SPI ROM, and decompresses it into DRAM
*   An ABL writes the APOB to DRAM  for consumption by the x86-based
AGESA
*   PSP releases the x86 processor from reset.  The x86 core fetches
and executes instructions from the reset vector

## Picasso Reset Vector and First Instructions

As mentioned above, prior to releasing the x86 main core from reset,
the PSP decompresses a BIOS image into DRAM.  The PSP uses a specific
BIOS Directory Table entry type to determine the source address (in
flash), the destination address (in DRAM), and the destination size.
The decompressed image is at the top of the destination region.  The
PSP then

Calculates the x86 reset vector as

    reset_vector	= dest_addr + dest_size - 0x10

Sets x86 CS descriptor shadow register to

    base		= dest_addr + dest_size - 0x10000
    limit		= 0xffff

Like all x86 devices, the main core is allowed to begin executing
instructions with

    CS:IP		= 0xf000:0xfff0

For example, assume the BIOS Directory Table indicates

    destination	= 0x9b00000
    size		= 0x300000

… then the BIOS image is placed at the topmost position the region
0x9b00000-0x9dfffff and

    reset_vector	= 0x9dffff0
    CS_shdw_base	= 0x9df0000
    CS:IP		= 0xf000:0xfff0

Although the x86 behaves as though it began executing at 0xfffffff0
i.e. 0xf000:0xfff0, the initial GDT load must use the physical address
of the table and not the typical CS-centric address.  And, the first
jump to protected mode must jump to the physical address in DRAM.  Any
code that is position-dependent must be linked to run at the final
destination.

## Implementation for coreboot

Supporting Picasso doesn’t fit perfectly with many of the coreboot
assumptions about x86 processors.  Changes are introduced primarily
into arch/x86 to accommodate a processor starting in DRAM and at a
nontraditional reset vector.

### CAR and early stages

The traditional coreboot bootblock and romstage rely on cache-as-RAM
and a linker script that positions temporary storage accordingly.  A
substitute for the DCACHE variables, called EARLYRAM, is introduced.
Like DCACHE, this allows for a consistent mapping of early regions
required across multiple stages prior to cbmem coming online.
Examples are the _preram_cbmem_console and _timestamp.

Due to Picasso's unique nature of starting with DRAM already available,
no early stages run as execute-in-place (XIP).  All post-bootblock
stages are copied from the BIOS flash into DRAM for faster
performance, and these regions are marked reserved later in POST.

Unlike CAR-based systems, and because Picasso does not run early
stages as XIP, its early stages are not constrained in their use
of .bss or .data sections.  All stages' .bss is zeroed, and all
.data sections are fully R/W at load time.

### bootblock

Picasso uses a bootblock that mirrors a traditional bootblock as much
as possible.  Because the image is loaded by the PSP, the bootblock is
not restricted to the top of the BIOS flash device.  The compressed
image is added into the PSP's `amdfw.rom` build.

### vboot app and verstage

Development is currently underway for the vboot app, and potentially
an x86-based verstage companion.  This document shall be updated once
the design is finalized and functioning.  Support for the PSP honoring
the presence of the vboot app is available only in certain SKUs.

### romstage and postcar

A traditional romstage is maintained for Picasso.  The primary reason for
this choice is to remain compatible with coreboot conventions and
to support the FSP 2.0 driver.  Picasso's romstage uses an
fsp_memory_init() call to glean the memory map from AGESA.  (See below.)
fsp_memory_init() brings cbmem online before returning to the caller.

No postcar stage is required or supported.

## AGESA v9 on Picasso

Due to the current restriction on publishing AGESA source, a pre-built
binary solution remains a requirement.  Modifying v9 to conform to the
existing v5 binaryPI interface was considered impractical.

Given the UEFI nature of modern AGESA, and the existing open source
work from Intel, Picasso shall support AGESA via an FSP-like prebuilt
image.  The Intel Firmware Support Package<sup>5</sup> combines
reference code with EDK II source to create a modular image with
discoverable entry points.  coreboot source already contains knowledge
of FSP, how to parse it, integrate it, and how to communicate with it.
Picasso's FSP is compatible with rev. 2.0 of the External Architecture
Specification.  Deviations, e.g., no FSP-T support, shall be published
in an Integration Guide.

## APCB setup

APCBs are used to provide the PSP with SPD information and optionally a set of
GPIOs to use for selecting which SPD to load. A list of APCB files should be
specified in `APCB_SOURCES`.

### Generating APCBs
If you have a template APCB file, the `apcb_edit` tool can be used to inject the
SPD and GPIOs used to select the correct slot.

## Footnotes

1. *AMD Platform Security Processor BIOS Architecture Design Guide
for AMD Family 17h Processors* (PID #55758) and *AMD Platform
Security Processor BIOS Architecture Design Guide* (PID #54267) for
earlier products
2. [PSP Integration](psp_integration.md)
3. [https://www.amd.com/system/files/TechDocs/44065_Arch2008.pdf](https://www.amd.com/system/files/TechDocs/44065_Arch2008.pdf)
4. [https://en.wikichip.org/wiki/amd/cores/picasso](https://en.wikichip.org/wiki/amd/cores/picasso)
5. [https://www.intel.com/content/www/us/en/intelligent-systems/intel-firmware-support-package/intel-fsp-overview.html](https://www.intel.com/content/www/us/en/intelligent-systems/intel-firmware-support-package/intel-fsp-overview.html)
