// -*- mode: C++; c-file-style: "cc-mode" -*-
//*************************************************************************
// DESCRIPTION: Verilator: DfgVertex sub-classes
//
// Code available from: https://verilator.org
//
//*************************************************************************
//
// Copyright 2003-2025 by Wilson Snyder. This program is free software; you
// can redistribute it and/or modify it under the terms of either the GNU
// Lesser General Public License Version 3 or the Perl Artistic License
// Version 2.0.
// SPDX-License-Identifier: LGPL-3.0-only OR Artistic-2.0
//
//*************************************************************************
//
// This is a data-flow graph based representation of combinational logic,
// the main difference from a V3Graph is that DfgVertex owns the storage
// of it's input edges (operands/sources/arguments), and can access each
// input edge directly by indexing, making modifications more efficient
// than the linked list based structures used by V3Graph.
//
// A bulk of the DfgVertex sub-types are generated by astgen, and are
// analogous to the corresponding AstNode sub-types.
//
// See also the internals documentation docs/internals.rst
//
//*************************************************************************

#ifndef VERILATOR_V3DFGVERTICES_H_
#define VERILATOR_V3DFGVERTICES_H_

#ifndef VERILATOR_V3DFG_H_
#error "Use V3Dfg.h as the include"
#include "V3Dfg.h"  // This helps code analysis tools pick up symbols in V3Dfg.h
#define VL_NOT_FINAL  // This #define fixes broken code folding in the CLion IDE
#endif

// === Abstract base node types (DfgVertex*) ===================================

class DfgVertexVar VL_NOT_FINAL : public DfgVertexUnary {
    AstVar* const m_varp;  // The AstVar associated with this vertex (not owned by this vertex)
    AstVarScope* const m_varScopep;  // The AstVarScope associated with this vertex (not owned)
    // Location of driver of this variable. Only used for converting back to Ast. Might be nullptr.
    FileLine* m_driverFileLine = nullptr;
    // If this DfgVertexVar is a synthesized temporary, this is the Var/VarScope it stands for.
    AstNode* m_tmpForp = nullptr;

    bool selfEquals(const DfgVertex& that) const final VL_MT_DISABLED;
    V3Hash selfHash() const final VL_MT_DISABLED;

public:
    inline DfgVertexVar(DfgGraph& dfg, VDfgType type, AstVar* varp);
    inline DfgVertexVar(DfgGraph& dfg, VDfgType type, AstVarScope* vscp);
    inline ~DfgVertexVar();
    ASTGEN_MEMBERS_DfgVertexVar;

    const std::string srcName(size_t) const override { return ""; }

    AstVar* varp() const { return m_varp; }
    AstVarScope* varScopep() const { return m_varScopep; }
    AstNode* nodep() const {
        return m_varScopep ? static_cast<AstNode*>(m_varScopep) : static_cast<AstNode*>(m_varp);
    }

    FileLine* driverFileLine() const { return m_driverFileLine; }
    void driverFileLine(FileLine* flp) { m_driverFileLine = flp; }

    AstNode* tmpForp() const { return m_tmpForp; }
    void tmpForp(AstNode* nodep) { m_tmpForp = nodep; }

    bool isDrivenFullyByDfg() const {
        return srcp() && !srcp()->is<DfgVertexSplice>() && !varp()->isForced()
               && !varp()->isSigUserRWPublic();
    }

    // Variable referenced via an AstVarXRef (hierarchical reference)
    bool hasXRefs() const { return nodep()->user1() & 0x03; }
    static void setHasRdXRefs(AstNode* nodep) { nodep->user1(nodep->user1() | 0x01); }
    static void setHasWrXRefs(AstNode* nodep) { nodep->user1(nodep->user1() | 0x02); }

    // Variable referenced from Ast code in the same module/netlist
    bool hasModRdRefs() const { return nodep()->user1() & 0x04; }
    bool hasModWrRefs() const { return nodep()->user1() & 0x08; }
    bool hasModRefs() const { return nodep()->user1() & 0x0c; }
    static void setHasModRdRefs(AstNode* nodep) { nodep->user1(nodep->user1() | 0x04); }
    static void setHasModWrRefs(AstNode* nodep) { nodep->user1(nodep->user1() | 0x08); }
    void setHasModRdRefs() const { setHasModRdRefs(nodep()); }
    void setHasModWrRefs() const { setHasModWrRefs(nodep()); }

    // Variable referenced from other DFG in the same module/netlist
    bool hasDfgRefs() const { return nodep()->user1() >> 5; }  // I.e.: (nodep()->user1() >> 4) > 1

    // Variable referenced outside the containing module/netlist.
    bool hasExtRefs() const {
        // In scoped mode, we can ignrore some of these as they were made explicit by then
        if (!m_varScopep) {
            if (m_varp->isIO()) return true;  // Ports
            if (m_varp->isTrace()) return true;  // Traced
            if (m_varp->isForced()) return true;  // Forced
            if (hasXRefs()) return true;  // Target of a hierarchical reference
        }
        if (m_varp->isPrimaryIO()) return true;  // Top level ports
        if (m_varp->isSigPublic()) return true;  // Public
        return false;
    }
};
class DfgVertexSplice VL_NOT_FINAL : public DfgVertexVariadic {
protected:
    struct DriverData final {
        FileLine* m_flp;  // Location of this driver
        uint32_t m_lo;  // Low index of range driven by this driver
        DriverData() = delete;
        DriverData(FileLine* flp, uint32_t lo)
            : m_flp{flp}
            , m_lo{lo} {}
    };
    std::vector<DriverData> m_driverData;  // Additional data associated with each driver

    bool selfEquals(const DfgVertex& that) const override VL_MT_DISABLED;
    V3Hash selfHash() const override VL_MT_DISABLED;

public:
    DfgVertexSplice(DfgGraph& dfg, VDfgType type, FileLine* flp, AstNodeDType* dtypep)
        : DfgVertexVariadic{dfg, type, flp, dtypep, 2u} {
        // Add optional source for 'defaultp'
        addSource();
    }
    ASTGEN_MEMBERS_DfgVertexSplice;

    std::pair<const DfgEdge*, size_t> sourceEdges() const override {
        const std::pair<const DfgEdge*, size_t> pair = DfgVertexVariadic::sourceEdges();
        UASSERT_OBJ(pair.second > 0, this, "default driver edge is missing");
        // If it has a default driver that's it
        if (pair.first->sourcep()) return pair;
        // Otherwise there is one less source
        return {pair.first + 1, pair.second - 1};
    }
    std::pair<DfgEdge*, size_t> sourceEdges() override {
        const auto pair = const_cast<const DfgVertexSplice*>(this)->sourceEdges();
        return {const_cast<DfgEdge*>(pair.first), pair.second};
    }

    // Named getter/setter for optional default driver
    DfgVertex* defaultp() const { return DfgVertexVariadic::source(0); }
    void defaultp(DfgVertex* vtxp) {
        UASSERT_OBJ(!vtxp->is<DfgLogic>(), vtxp, "default driver can't be a DfgLogic");
        const bool found = findSourceEdge([vtxp](const DfgEdge& e, size_t) -> bool {  //
            return e.sourcep() == vtxp;
        });
        UASSERT_OBJ(!found, this, "adding existing driver as default");
        DfgVertexVariadic::sourceEdge(0)->relinkSource(vtxp);
    }

    // Add resolved driver
    void addDriver(FileLine* flp, uint32_t lo, DfgVertex* vtxp) {
        UASSERT_OBJ(!vtxp->is<DfgLogic>(), vtxp, "addDriver called with DfgLogic");
        UASSERT_OBJ(vtxp != defaultp(), this, "adding default driver as resolved");
        m_driverData.emplace_back(flp, lo);
        DfgVertexVariadic::addSource()->relinkSource(vtxp);
    }

    FileLine* driverFileLine(size_t idx) const {
        UASSERT_OBJ(!defaultp() || idx > 0, this, "'driverFileLine' called on default driver");
        if (defaultp()) --idx;
        return m_driverData.at(idx).m_flp;
    }

    uint32_t driverLo(size_t idx) const {
        UASSERT_OBJ(!defaultp() || idx > 0, this, "'driverLo' called on default driver");
        if (defaultp()) --idx;
        const DriverData& dd = m_driverData.at(idx);
        return dd.m_lo;
    }

    DfgVertex* driverAt(size_t idx) const {
        const DfgEdge* const edgep = findSourceEdge([this, idx](const DfgEdge& e, size_t i) {  //
            // Don't pick the default driver
            if (i == 0 && defaultp()) return false;
            return driverLo(i) == idx;
        });
        return edgep ? edgep->sourcep() : nullptr;
    }

    // If drives the whole result explicitly (not through defaultp), this is
    // the actual driver this DfgVertexSplice can be replaced with.
    inline DfgVertex* wholep() const;

    // cppcheck-suppress duplInheritedMember
    void resetSources() {
        m_driverData.clear();
        // Unlink default driver
        DfgVertex* const dp = defaultp();
        DfgVertexVariadic::sourceEdge(0)->unlinkSource();
        // Reset DfgVertexVariadic sources
        DfgVertexVariadic::resetSources();
        // Add back the default driver if present
        DfgEdge* const edgep = DfgVertexVariadic::addSource();
        if (dp) edgep->relinkSource(dp);
    }

    const std::string srcName(size_t idx) const override {
        if (idx == 0 && defaultp()) return "default";
        const uint32_t lo = driverLo(idx);
        const uint32_t hi = lo + DfgVertexVariadic::source(idx + !defaultp())->size() - 1;
        return '[' + std::to_string(hi) + ':' + std::to_string(lo) + ']';
    }
};

// === Concrete node types =====================================================

// === DfgVertex ===
class DfgConst final : public DfgVertex {
    friend class DfgVertex;
    friend class DfgVisitor;

    V3Number m_num;  // Constant value

    bool selfEquals(const DfgVertex& that) const override VL_MT_DISABLED;
    V3Hash selfHash() const override VL_MT_DISABLED;

public:
    inline DfgConst(DfgGraph& dfg, FileLine* flp, const V3Number& num);
    inline DfgConst(DfgGraph& dfg, FileLine* flp, uint32_t width, uint32_t value = 0);
    ASTGEN_MEMBERS_DfgConst;

    V3Number& num() { return m_num; }
    const V3Number& num() const { return m_num; }

    size_t toSizeT() const {
        if VL_CONSTEXPR_CXX17 (sizeof(size_t) > sizeof(uint32_t)) {
            return static_cast<size_t>(num().toUQuad());
        }
        return static_cast<size_t>(num().toUInt());
    }

    uint32_t toU32() const { return static_cast<size_t>(num().toUInt()); }

    // cppcheck-suppress duplInheritedMember
    bool isZero() const { return num().isEqZero(); }
    // cppcheck-suppress duplInheritedMember
    bool isOnes() const { return num().isEqAllOnes(width()); }

    // Does this DfgConst have the given value? Note this is not easy to answer if wider than 32.
    bool hasValue(uint32_t value) const {
        return !num().isFourState() && num().edataWord(0) == value && num().mostSetBitP1() <= 32;
    }

    std::pair<DfgEdge*, size_t> sourceEdges() override { return {nullptr, 0}; }
    std::pair<const DfgEdge*, size_t> sourceEdges() const override { return {nullptr, 0}; }
    const string srcName(size_t) const override {  // LCOV_EXCL_START
        VL_UNREACHABLE;
        return "";
    }  // LCOV_EXCL_STOP
};

// === DfgVertexBinary ===
class DfgMux final : public DfgVertexBinary {
    // AstSel is ternary, but the 'widthp' is always constant and is hence redundant, and
    // 'lsbp' is very often constant. As AstSel is fairly common, we special case as a DfgSel for
    // the constant 'lsbp', and as 'DfgMux` for the non-constant 'lsbp'.
public:
    DfgMux(DfgGraph& dfg, FileLine* flp, AstNodeDType* dtypep)
        : DfgVertexBinary{dfg, dfgType(), flp, dtypep} {}
    ASTGEN_MEMBERS_DfgMux;

    DfgVertex* fromp() const { return source<0>(); }
    void fromp(DfgVertex* vtxp) { relinkSource<0>(vtxp); }
    DfgVertex* lsbp() const { return source<1>(); }
    void lsbp(DfgVertex* vtxp) { relinkSource<1>(vtxp); }

    const string srcName(size_t idx) const override { return idx ? "lsbp" : "fromp"; }
};

// === DfgVertexUnary ===

class DfgSel final : public DfgVertexUnary {
    // AstSel is ternary, but the 'widthp' is always constant and is hence redundant, and
    // 'lsbp' is very often constant. As AstSel is fairly common, we special case as a DfgSel for
    // the constant 'lsbp', and as 'DfgMux` for the non-constant 'lsbp'.
    uint32_t m_lsb = 0;  // The LSB index

    bool selfEquals(const DfgVertex& that) const override VL_MT_DISABLED;
    V3Hash selfHash() const override VL_MT_DISABLED;

public:
    DfgSel(DfgGraph& dfg, FileLine* flp, AstNodeDType* dtypep)
        : DfgVertexUnary{dfg, dfgType(), flp, dtypep} {}
    ASTGEN_MEMBERS_DfgSel;

    DfgVertex* fromp() const { return source<0>(); }
    void fromp(DfgVertex* vtxp) { relinkSource<0>(vtxp); }
    uint32_t lsb() const { return m_lsb; }
    void lsb(uint32_t value) { m_lsb = value; }

    const string srcName(size_t) const override { return "fromp"; }
};

class DfgUnitArray final : public DfgVertexUnary {
    // This is a type adapter for modeling arrays. It's a single element array,
    // with the value of the single element being the source operand.
public:
    DfgUnitArray(DfgGraph& dfg, FileLine* flp, AstNodeDType* dtypep)
        : DfgVertexUnary{dfg, dfgType(), flp, dtypep} {
        UASSERT_OBJ(this->dtypep(), flp, "Non array DfgUnitArray");
        UASSERT_OBJ(this->size() == 1, flp, "DfgUnitArray must have a single element");
    }
    ASTGEN_MEMBERS_DfgUnitArray;

    const std::string srcName(size_t) const override { return ""; }
};

// === DfgVertexVar ===
class DfgVarArray final : public DfgVertexVar {
    friend class DfgVertex;
    friend class DfgVisitor;

public:
    DfgVarArray(DfgGraph& dfg, AstVar* varp)
        : DfgVertexVar{dfg, dfgType(), varp} {
        UASSERT_OBJ(VN_IS(dtypep(), UnpackArrayDType), varp, "Non array DfgVarArray");
    }
    DfgVarArray(DfgGraph& dfg, AstVarScope* vscp)
        : DfgVertexVar{dfg, dfgType(), vscp} {
        UASSERT_OBJ(VN_IS(dtypep(), UnpackArrayDType), vscp, "Non array DfgVarArray");
    }
    ASTGEN_MEMBERS_DfgVarArray;
};
class DfgVarPacked final : public DfgVertexVar {
    friend class DfgVertex;
    friend class DfgVisitor;

public:
    DfgVarPacked(DfgGraph& dfg, AstVar* varp)
        : DfgVertexVar{dfg, dfgType(), varp} {
        UASSERT_OBJ(!VN_IS(dtypep(), UnpackArrayDType), varp, "Array DfgVarPacked");
    }
    DfgVarPacked(DfgGraph& dfg, AstVarScope* vscp)
        : DfgVertexVar{dfg, dfgType(), vscp} {
        UASSERT_OBJ(!VN_IS(dtypep(), UnpackArrayDType), vscp, "Array DfgVarPacked");
    }
    ASTGEN_MEMBERS_DfgVarPacked;
};

// === DfgVertexVariadic ===
class DfgLogic final : public DfgVertexVariadic {
    // Generic vertex representing a whole combinational process
    AstNode* const m_nodep;  // The Ast logic represented by this vertex
    const std::unique_ptr<CfgGraph> m_cfgp;
    // Vertices this logic was synthesized into. Excluding variables
    std::vector<DfgVertex*> m_synth;

public:
    DfgLogic(DfgGraph& dfg, AstAssignW* nodep)
        : DfgVertexVariadic{dfg, dfgType(), nodep->fileline(), nullptr, 1u}
        , m_nodep{nodep}
        , m_cfgp{nullptr} {}

    DfgLogic(DfgGraph& dfg, AstAlways* nodep, std::unique_ptr<CfgGraph> cfgp)
        : DfgVertexVariadic{dfg, dfgType(), nodep->fileline(), nullptr, 1u}
        , m_nodep{nodep}
        , m_cfgp{std::move(cfgp)} {}

    ASTGEN_MEMBERS_DfgLogic;

    void addInput(DfgVertexVar* varp) { addSource()->relinkSource(varp); }

    AstNode* nodep() const { return m_nodep; }
    CfgGraph& cfg() { return *m_cfgp; }
    const CfgGraph& cfg() const { return *m_cfgp; }
    std::vector<DfgVertex*>& synth() { return m_synth; }
    const std::vector<DfgVertex*>& synth() const { return m_synth; }

    const std::string srcName(size_t) const override { return ""; }
};

class DfgUnresolved final : public DfgVertexVariadic {
    // Represents a collection of unresolved variable drivers before synthesis

public:
    DfgUnresolved(DfgGraph& dfg, const DfgVertexVar* vtxp)
        : DfgVertexVariadic{dfg, dfgType(), vtxp->fileline(), vtxp->dtypep(), 1u} {}
    ASTGEN_MEMBERS_DfgUnresolved;

    // Can only be driven by DfgLogic or DfgVertexSplice
    void addDriver(DfgLogic* vtxp) { addSource()->relinkSource(vtxp); }
    void addDriver(DfgVertexSplice* vtxp) { addSource()->relinkSource(vtxp); }

    // cppcheck-suppress duplInheritedMember
    void clearSources() { DfgVertexVariadic::clearSources(); }

    DfgVertex* singleSource() const { return arity() == 1 ? source(0) : nullptr; }

    const std::string srcName(size_t) const override { return ""; }
};

// === DfgVertexSplice ===
class DfgSpliceArray final : public DfgVertexSplice {
    friend class DfgVertex;
    friend class DfgVisitor;

public:
    DfgSpliceArray(DfgGraph& dfg, FileLine* flp, AstNodeDType* dtypep)
        : DfgVertexSplice{dfg, dfgType(), flp, dtypep} {
        UASSERT_OBJ(VN_IS(dtypep, UnpackArrayDType), flp, "Non array DfgSpliceArray");
    }
    ASTGEN_MEMBERS_DfgSpliceArray;
};
class DfgSplicePacked final : public DfgVertexSplice {
    friend class DfgVertex;
    friend class DfgVisitor;

public:
    DfgSplicePacked(DfgGraph& dfg, FileLine* flp, AstNodeDType* dtypep)
        : DfgVertexSplice{dfg, dfgType(), flp, dtypep} {
        UASSERT_OBJ(!VN_IS(dtypep, UnpackArrayDType), flp, "Array DfgSplicePacked");
    }
    ASTGEN_MEMBERS_DfgSplicePacked;
};

#endif
