// 

/**************************************************************************\
 * Copyright (c) Kongsberg Oil & Gas Technologies AS
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
\**************************************************************************/

// Pulls in __COIN_SOQT__ define used below.
#include <Inventor/Qt/SoQtBasic.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif // HAVE_CONFIG_H

#include <soqtdefs.h>
#include <Inventor/Qt/SoQtObject.h>
#include <Inventor/Qt/devices/SoQtDevice.h>
#include <Inventor/Qt/SoQtComponent.h>

/*!
  \class SoQtObject Inventor/Qt/SoQtObject.h
  \brief The SoQtObject class is the common superclass for all SoQt component classes.

  \ingroup misc

  The purpose of making this class the superclass of all SoQt
  device, component and viewer classes is to be able to do runtime
  type checking of the SoQt objects.


  You can place the macro SOQT_OBJECT_HEADER(classname,parentname)
  within a class definition header for SoQt extension components to
  automatically provide the necessary definitions for setting up a
  runtime type system for your extension classes:

  \code
  #ifndef MYSPECIALVIEWER_H
  #define MYSPECIALVIEWER_H

  class MySpecialViewer : public SoQtExaminerViewer {
    SOQT_OBJECT_HEADER(MySpecialViewer, SoQtExaminerViewer);
    // [rest of class definition follows]
  };

  #endif // !MYSPECIALVIEWER_H
  \endcode

  Then put the SOQT_OBJECT_SOURCE(classname) macro within the
  actual implementation source code file to include the necessary
  code for the runtime type system:

  \code
  #include <MySpecialViewer.h>

  SOQT_OBJECT_SOURCE(MySpecialViewer);

  // [rest of class implementation]
  \endcode


  See also the documentation of the SoType class in Coin or Inventor.
 */

// *************************************************************************

/*!
  \fn SoType SoQtObject::getTypeId

  Returns the type identification of an object derived from a class
  inheriting SoQtObject.  This is used for runtime type checking
  and "downward" casting.

  Usage example:

  \code
  void foo(SoQtViewer * comp)
  {
    if (comp->getTypeId() == SoQtExaminerViewer::getClassTypeId()) {
      // safe downward cast, knows the type
      SoQtExaminerViewer * exviewer = (SoQtExaminerViewer *)comp;
      /// [then something] ///
    }
    else if (comp->getTypeId().isOfType(SoQtFlyViewer::getClassTypeId())) {
      // safe downward cast, knows the type
      SoQtFlyViewer * flyviewer = (SoQtFlyViewer *)comp;
      // then something else
    }
  }
  \endcode
*/
// FIXME: add doc above to explain how external developers can use the
// type system for their own extension classes. 20020502 mortene.

// This is a private variable.
SoType SoQtObject::classTypeId SOQT_STATIC_SOTYPE_INIT;

/*!
  Sets up initialization for data common to all instances of this
  class, submitting necessary information to the internal SoQt type
  system.
*/
void
SoQtObject::initClass(void)
{
  assert(SoQtObject::classTypeId == SoType::badType());
  SoQtObject::classTypeId =
    SoType::createType(SoType::badType(), "SoQtObject");
}

/*!
  Returns \c TRUE if the type of this object is either of the same
  type or inherited from \a type.
*/
SbBool
SoQtObject::isOfType(SoType type) const
{
  return this->getTypeId().isDerivedFrom(type);
}

/*!
  This static method returns the SoType object associated with
  objects of this class.
*/
SoType
SoQtObject::getClassTypeId(void)
{
  return SoQtObject::classTypeId;
}

/*!
  Initialize the type system of SoQtObject, all SoQt device
  classes and all SoQt components (including viewers).
*/
void
SoQtObject::init(void)
{
  SoQtObject::initClass();
  SoQtDevice::initClasses();
  SoQtComponent::initClasses();
}
