package org.apache.stratum.scheduler;

/*
 * Copyright 2001-2005 The Apache Software Foundation or its licensors,
 * as applicable.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.FileInputStream;
import java.util.List;
import java.util.Properties;

import org.apache.commons.betwixt.XMLIntrospector;
import org.apache.commons.betwixt.io.BeanReader;
import org.apache.commons.betwixt.strategy.DecapitalizeNameMapper;
import org.apache.commons.betwixt.strategy.DefaultPluralStemmer;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.lang.exception.NestableException;
import org.apache.log4j.Category;
import org.apache.stratum.lifecycle.Configurable;
import org.apache.stratum.lifecycle.Startable;

import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.SchedulerException;
import org.quartz.impl.StdScheduler;
import org.quartz.impl.StdSchedulerFactory;

/**
 * This class is the Scheduler component that implements a Quartz scheduler.
 *
 * @author <a href="mailto:john@zenplex.com">John Thorhauer</a>
 * @version $Id: Scheduler.java 264731 2005-08-30 08:04:32Z henning $
 */
public class Scheduler
        implements Configurable, Startable
{
    /** Log4j category used for logging. */
    private static Category log = Category.getInstance(Scheduler.class);

    /** TODO: DOCUMENT ME! */
    private static String confLocation = "/projects/jakarta-turbine-stratum/src/test-conf/Scheduler.properties";

    /** TODO: DOCUMENT ME! */
    private StdSchedulerFactory schedulerFactory;

    /** TODO: DOCUMENT ME! */
    protected StdScheduler scheduler;

    /**
     * Just a command line test tool to start the scheduler from the command line.  If looks for the properties file in the
     * following location: /projects/jakarta-turbine-stratum/src/test-conf/Scheduler.properties
     *
     * @param args command line arguments
     *
     * @throws Exception
     */
    public static void main(String [] args)
            throws Exception
    {
        Scheduler sched = new Scheduler();
        PropertiesConfiguration conf = new PropertiesConfiguration(confLocation);
        sched.configure(conf);
        sched.start();
    }

    /**
     * start the scheduler.
     *
     * @throws Exception
     */
    public void start()
            throws Exception
    {
        scheduler.start();
    }

    /**
     * stop the scheduler.
     *
     * @throws Exception
     */
    public void stop()
            throws Exception
    {
        scheduler.shutdown();
    }

    /**
     * Configure the Scheduler
     *
     * @param configuration the configuration
     *
     * @throws NestableException
     */
    public void configure(Configuration configuration)
            throws NestableException
    {
        String xmlLocation = configuration.getString("scheduler.config.location");

        try
        {
            // get scheduler factory and initialize it with
            // settings from the .xml file
            SchedulerConfig schedConfig = getSchedulerConfig(xmlLocation);
            Properties factoryProps;

            try
            {
                factoryProps = getFactoryProps(schedConfig);
            }
            catch (NullPointerException ex)
            {
                log.error("Error loading properties to initialize"
                        + " Scheduler Factory. Please make sure the following"
                        + " elements are properly filled out in the"
                        + " Scheduler .xml config file:\n"
                        + "   <instanceName></instanceName>\n"
                        + "    <threadPoolConfig>\n"
                        + "     <className></className>\n"
                        + "     <threadCount></threadCount>\n"
                        + "     <threadPriority></threadPriority>\n"
                        + "   </threadPoolConfig>\n"
                        + "   <jobStoreConfig>\n"
                        + "     <className></className>\n"
                        + "   </jobStoreConfig>");

                return;
            }

            schedulerFactory = new StdSchedulerFactory();
            schedulerFactory.initialize(factoryProps);

            scheduler = (StdScheduler) schedulerFactory.getScheduler();

            List jobConfigs = schedConfig.getJobConfigs();

            if (!jobConfigs.isEmpty())
            {
                for (int i = 0; i < jobConfigs.size(); i++)
                {
                    JobConfig jobConf = (JobConfig) jobConfigs.get(i);
                    String jobName = (String) jobConf.getName();

                    String jobGroup = jobConf.getGroup();
                    String jobClassName = jobConf.getClassName();

                    JobDetail job = new JobDetail(jobName, jobGroup, Class.forName(jobClassName));

                    //get the trigger for this job if it exists
                    TriggerConfig triggerConf = getTriggerConfig(schedConfig, job.getName());

                    if (triggerConf != null)
                    {
                        job.getName();

                        String triggerGroup = triggerConf.getGroup();
                        String triggerName = triggerConf.getName();
                        String cronExpression =
                            triggerConf.getSeconds() + " " + triggerConf.getMinutes() + " " + triggerConf.getHours() + " "
                            + triggerConf.getDayOfMonth() + " " + triggerConf.getMonth() + " " + triggerConf.getDayOfWeek();

                        CronTrigger cronTrigger = new CronTrigger(triggerName, triggerGroup, jobName, jobGroup, cronExpression);

                        scheduler.scheduleJob(job, cronTrigger);
                        triggerConf = null;
                        cronTrigger = null;
                    }
                }
            }
        }
        catch (SchedulerException ex)
        {
            log.error("Error Initializing Scheduler:  " + ex.toString(), ex);
        }
        catch (ClassNotFoundException ex)
        {
            log.error("Error Loading class:  " + ex.toString(), ex);
        }
        catch (Exception ex)
        {
            log.error("Error:  " + ex.toString(), ex);
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param schdConf Scheduler Config
     *
     * @return TODO: DOCUMENT ME!
     */
    private Properties getFactoryProps(SchedulerConfig schdConf)
    {
        Properties props = new Properties();

        props.put("org.quartz.scheduler.instanceName", schdConf.getInstanceName());

        ThreadPoolConfig threadPool = schdConf.getThreadPoolConfig();
        props.put("org.quartz.threadPool.class", threadPool.getClassName());
        props.put("org.quartz.threadPool.threadCount", threadPool.getThreadCount());
        props.put("org.quartz.threadPool.threadPriority", threadPool.getThreadPriority());

        JobStoreConfig jobStore = schdConf.getJobStoreConfig();
        props.put("org.quartz.jobStore.class", jobStore.getClassName());

        return props;
    }

    /**
     * Loads the SchedulerConfig object from the xml file
     *
     * @param xmlLocation of the xml file
     *
     * @return TODO: DOCUMENT ME!
     *
     * @throws Exception
     */
    private SchedulerConfig getSchedulerConfig(String xmlLocation)
            throws Exception
    {
        FileInputStream in = new FileInputStream(xmlLocation);

        // create a new BeanReader
        BeanReader reader = createBeanReader();

        SchedulerConfig schedConf = (SchedulerConfig) reader.parse(in);

        return schedConf;
    }

    /**
     * Creates a beanreader
     *
     * @return beanreader
     *
     * @throws Exception
     */
    private BeanReader createBeanReader()
            throws Exception
    {
        BeanReader reader = new BeanReader();
        reader.setXMLIntrospector(createXMLIntrospector());
        reader.registerBeanClass(SchedulerConfig.class);

        return reader;
    }

    /**
     * Loads Xml Introspector with needed values
     *
     * @return introspector
     */
    private XMLIntrospector createXMLIntrospector()
    {
        XMLIntrospector introspector = new XMLIntrospector();

        // set elements for attributes to true
        introspector.getConfiguration().setAttributesForPrimitives(false);

        // wrap collections in an XML element
        //introspector.setWrapCollectionsInElement(true);
        // turn bean elements first letter into lower case
        introspector.getConfiguration().setElementNameMapper(new DecapitalizeNameMapper());
        introspector.getConfiguration().setAttributeNameMapper(new DecapitalizeNameMapper());

        introspector.getConfiguration().setPluralStemmer(new DefaultPluralStemmer());

        return introspector;
    }

    private TriggerConfig getTriggerConfig(SchedulerConfig schedConf, String jobName)
    {
        TriggerConfig trig = null;
        List triggers = schedConf.getTriggerConfigs();

        for (int i = 0; i < triggers.size(); i++)
        {
            TriggerConfig tmpTrig = (TriggerConfig) triggers.get(i);

            if (tmpTrig.getJobName().equals(jobName))
            {
                trig = tmpTrig;

                return trig;
            }
        }

        return trig;
    }
}
