/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.mirror;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.header.ConnectHeaders;
import org.apache.kafka.connect.header.Headers;
import org.apache.kafka.connect.mirror.MirrorSourceConnector;
import org.apache.kafka.connect.mirror.MirrorSourceMetrics;
import org.apache.kafka.connect.mirror.MirrorSourceTaskConfig;
import org.apache.kafka.connect.mirror.MirrorUtils;
import org.apache.kafka.connect.mirror.OffsetSync;
import org.apache.kafka.connect.mirror.ReplicationPolicy;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MirrorSourceTask
extends SourceTask {
    private static final Logger log = LoggerFactory.getLogger(MirrorSourceTask.class);
    private static final int MAX_OUTSTANDING_OFFSET_SYNCS = 10;
    private KafkaConsumer<byte[], byte[]> consumer;
    private KafkaProducer<byte[], byte[]> offsetProducer;
    private String sourceClusterAlias;
    private String offsetSyncsTopic;
    private Duration pollTimeout;
    private long maxOffsetLag;
    private Map<TopicPartition, PartitionState> partitionStates;
    private ReplicationPolicy replicationPolicy;
    private MirrorSourceMetrics metrics;
    private boolean stopping = false;
    private final Map<TopicPartition, OffsetSync> delayedOffsetSyncs = new LinkedHashMap<TopicPartition, OffsetSync>();
    private final Map<TopicPartition, OffsetSync> pendingOffsetSyncs = new LinkedHashMap<TopicPartition, OffsetSync>();
    private Semaphore outstandingOffsetSyncs;
    private Semaphore consumerAccess;

    public MirrorSourceTask() {
    }

    MirrorSourceTask(KafkaConsumer<byte[], byte[]> consumer, MirrorSourceMetrics metrics, String sourceClusterAlias, ReplicationPolicy replicationPolicy, long maxOffsetLag, KafkaProducer<byte[], byte[]> producer, Semaphore outstandingOffsetSyncs, Map<TopicPartition, PartitionState> partitionStates, String offsetSyncsTopic) {
        this.consumer = consumer;
        this.metrics = metrics;
        this.sourceClusterAlias = sourceClusterAlias;
        this.replicationPolicy = replicationPolicy;
        this.maxOffsetLag = maxOffsetLag;
        this.consumerAccess = new Semaphore(1);
        this.offsetProducer = producer;
        this.outstandingOffsetSyncs = outstandingOffsetSyncs;
        this.partitionStates = partitionStates;
        this.offsetSyncsTopic = offsetSyncsTopic;
    }

    public void start(Map<String, String> props) {
        MirrorSourceTaskConfig config = new MirrorSourceTaskConfig(props);
        this.pendingOffsetSyncs.clear();
        this.outstandingOffsetSyncs = new Semaphore(10);
        this.consumerAccess = new Semaphore(1);
        this.sourceClusterAlias = config.sourceClusterAlias();
        this.metrics = config.metrics();
        this.pollTimeout = config.consumerPollTimeout();
        this.maxOffsetLag = config.maxOffsetLag();
        this.replicationPolicy = config.replicationPolicy();
        this.partitionStates = new HashMap<TopicPartition, PartitionState>();
        this.offsetSyncsTopic = config.offsetSyncsTopic();
        this.consumer = MirrorUtils.newConsumer(config.sourceConsumerConfig("replication-consumer"));
        this.offsetProducer = MirrorUtils.newProducer(config.offsetSyncsTopicProducerConfig());
        Set<TopicPartition> taskTopicPartitions = config.taskTopicPartitions();
        this.initializeConsumer(taskTopicPartitions);
        log.info("{} replicating {} topic-partitions {}->{}: {}.", new Object[]{Thread.currentThread().getName(), taskTopicPartitions.size(), this.sourceClusterAlias, config.targetClusterAlias(), taskTopicPartitions});
    }

    public void commit() {
        this.promoteDelayedOffsetSyncs();
        this.firePendingOffsetSyncs();
    }

    public void stop() {
        long start = System.currentTimeMillis();
        this.stopping = true;
        this.consumer.wakeup();
        try {
            this.consumerAccess.acquire();
        }
        catch (InterruptedException e) {
            log.warn("Interrupted waiting for access to consumer. Will try closing anyway.");
        }
        Utils.closeQuietly(this.consumer, (String)"source consumer");
        Utils.closeQuietly(this.offsetProducer, (String)"offset producer");
        Utils.closeQuietly((AutoCloseable)this.metrics, (String)"metrics");
        log.info("Stopping {} took {} ms.", (Object)Thread.currentThread().getName(), (Object)(System.currentTimeMillis() - start));
    }

    public String version() {
        return new MirrorSourceConnector().version();
    }

    public List<SourceRecord> poll() {
        List<SourceRecord> list;
        if (!this.consumerAccess.tryAcquire()) {
            return null;
        }
        if (this.stopping) {
            return null;
        }
        try {
            Object object;
            ConsumerRecords records = this.consumer.poll(this.pollTimeout);
            ArrayList<SourceRecord> sourceRecords = new ArrayList<SourceRecord>(records.count());
            for (ConsumerRecord record : records) {
                SourceRecord converted = this.convertRecord((ConsumerRecord<byte[], byte[]>)record);
                sourceRecords.add(converted);
                TopicPartition topicPartition = new TopicPartition(converted.topic(), converted.kafkaPartition().intValue());
                this.metrics.recordAge(topicPartition, System.currentTimeMillis() - record.timestamp());
                this.metrics.recordBytes(topicPartition, MirrorSourceTask.byteSize((byte[])record.value()));
            }
            if (sourceRecords.isEmpty()) {
                object = null;
                return object;
            }
            log.trace("Polled {} records from {}.", (Object)sourceRecords.size(), (Object)records.partitions());
            object = sourceRecords;
            return object;
        }
        catch (WakeupException e) {
            list = null;
            return list;
        }
        catch (KafkaException e) {
            log.warn("Failure during poll.", (Throwable)e);
            list = null;
            return list;
        }
        catch (Throwable e) {
            log.error("Failure during poll.", e);
            throw e;
        }
        finally {
            this.consumerAccess.release();
        }
    }

    public void commitRecord(SourceRecord record, RecordMetadata metadata) {
        if (this.stopping) {
            return;
        }
        if (metadata == null) {
            log.debug("No RecordMetadata (source record was probably filtered out during transformation) -- can't sync offsets for {}.", (Object)record.topic());
            return;
        }
        if (!metadata.hasOffset()) {
            log.error("RecordMetadata has no offset -- can't sync offsets for {}.", (Object)record.topic());
            return;
        }
        TopicPartition topicPartition = new TopicPartition(record.topic(), record.kafkaPartition().intValue());
        long latency = System.currentTimeMillis() - record.timestamp();
        this.metrics.countRecord(topicPartition);
        this.metrics.replicationLatency(topicPartition, latency);
        TopicPartition sourceTopicPartition = MirrorUtils.unwrapPartition(record.sourcePartition());
        long upstreamOffset = MirrorUtils.unwrapOffset(record.sourceOffset());
        long downstreamOffset = metadata.offset();
        this.maybeQueueOffsetSyncs(sourceTopicPartition, upstreamOffset, downstreamOffset);
        this.firePendingOffsetSyncs();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeQueueOffsetSyncs(TopicPartition topicPartition, long upstreamOffset, long downstreamOffset) {
        PartitionState partitionState = this.partitionStates.computeIfAbsent(topicPartition, x -> new PartitionState(this.maxOffsetLag));
        OffsetSync offsetSync = new OffsetSync(topicPartition, upstreamOffset, downstreamOffset);
        if (partitionState.update(upstreamOffset, downstreamOffset)) {
            MirrorSourceTask mirrorSourceTask = this;
            synchronized (mirrorSourceTask) {
                this.delayedOffsetSyncs.remove(topicPartition);
                this.pendingOffsetSyncs.put(topicPartition, offsetSync);
            }
            partitionState.reset();
        } else {
            MirrorSourceTask mirrorSourceTask = this;
            synchronized (mirrorSourceTask) {
                this.delayedOffsetSyncs.put(topicPartition, offsetSync);
            }
        }
    }

    private synchronized void promoteDelayedOffsetSyncs() {
        this.pendingOffsetSyncs.putAll(this.delayedOffsetSyncs);
        this.delayedOffsetSyncs.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void firePendingOffsetSyncs() {
        while (true) {
            OffsetSync pendingOffsetSync;
            MirrorSourceTask mirrorSourceTask = this;
            synchronized (mirrorSourceTask) {
                Iterator<OffsetSync> syncIterator = this.pendingOffsetSyncs.values().iterator();
                if (!syncIterator.hasNext()) {
                    log.trace("No more pending offset syncs");
                    return;
                }
                pendingOffsetSync = syncIterator.next();
                if (!this.outstandingOffsetSyncs.tryAcquire()) {
                    log.trace("Too many in-flight offset syncs; will try to send remaining offset syncs later");
                    return;
                }
                syncIterator.remove();
            }
            this.sendOffsetSync(pendingOffsetSync);
            log.trace("Dispatched offset sync for {}", (Object)pendingOffsetSync.topicPartition());
        }
    }

    private void sendOffsetSync(OffsetSync offsetSync) {
        ProducerRecord record = new ProducerRecord(this.offsetSyncsTopic, Integer.valueOf(0), (Object)offsetSync.recordKey(), (Object)offsetSync.recordValue());
        this.offsetProducer.send(record, (x, e) -> {
            if (e != null) {
                log.error("Failure sending offset sync.", (Throwable)e);
            } else {
                log.trace("Sync'd offsets for {}: {}=={}", new Object[]{offsetSync.topicPartition(), offsetSync.upstreamOffset(), offsetSync.downstreamOffset()});
            }
            this.outstandingOffsetSyncs.release();
        });
    }

    private Map<TopicPartition, Long> loadOffsets(Set<TopicPartition> topicPartitions) {
        return topicPartitions.stream().collect(Collectors.toMap(x -> x, this::loadOffset));
    }

    private Long loadOffset(TopicPartition topicPartition) {
        Map<String, Object> wrappedPartition = MirrorUtils.wrapPartition(topicPartition, this.sourceClusterAlias);
        Map wrappedOffset = this.context.offsetStorageReader().offset(wrappedPartition);
        return MirrorUtils.unwrapOffset(wrappedOffset);
    }

    void initializeConsumer(Set<TopicPartition> taskTopicPartitions) {
        Map<TopicPartition, Long> topicPartitionOffsets = this.loadOffsets(taskTopicPartitions);
        this.consumer.assign(topicPartitionOffsets.keySet());
        log.info("Starting with {} previously uncommitted partitions.", (Object)topicPartitionOffsets.values().stream().filter(this::isUncommitted).count());
        topicPartitionOffsets.forEach((topicPartition, offset) -> {
            if (this.isUncommitted((Long)offset)) {
                log.trace("Skipping seeking offset for topicPartition: {}", topicPartition);
                return;
            }
            long nextOffsetToCommittedOffset = offset + 1L;
            log.trace("Seeking to offset {} for topicPartition: {}", (Object)nextOffsetToCommittedOffset, topicPartition);
            this.consumer.seek(topicPartition, nextOffsetToCommittedOffset);
        });
    }

    SourceRecord convertRecord(ConsumerRecord<byte[], byte[]> record) {
        String targetTopic = this.formatRemoteTopic(record.topic());
        Headers headers = this.convertHeaders(record);
        return new SourceRecord(MirrorUtils.wrapPartition(new TopicPartition(record.topic(), record.partition()), this.sourceClusterAlias), MirrorUtils.wrapOffset(record.offset()), targetTopic, Integer.valueOf(record.partition()), Schema.OPTIONAL_BYTES_SCHEMA, record.key(), Schema.BYTES_SCHEMA, record.value(), Long.valueOf(record.timestamp()), (Iterable)headers);
    }

    private Headers convertHeaders(ConsumerRecord<byte[], byte[]> record) {
        ConnectHeaders headers = new ConnectHeaders();
        for (Header header : record.headers()) {
            headers.addBytes(header.key(), header.value());
        }
        return headers;
    }

    private String formatRemoteTopic(String topic) {
        return this.replicationPolicy.formatRemoteTopic(this.sourceClusterAlias, topic);
    }

    private static int byteSize(byte[] bytes) {
        if (bytes == null) {
            return 0;
        }
        return bytes.length;
    }

    private boolean isUncommitted(Long offset) {
        return offset == null || offset < 0L;
    }

    static class PartitionState {
        long previousUpstreamOffset = -1L;
        long previousDownstreamOffset = -1L;
        long lastSyncDownstreamOffset = -1L;
        long maxOffsetLag;
        boolean shouldSyncOffsets;

        PartitionState(long maxOffsetLag) {
            this.maxOffsetLag = maxOffsetLag;
        }

        boolean update(long upstreamOffset, long downstreamOffset) {
            boolean truncatedDownstreamTopic;
            boolean noPreviousSyncThisLifetime = this.lastSyncDownstreamOffset == -1L;
            boolean translatedOffsetTooStale = downstreamOffset - (this.lastSyncDownstreamOffset + 1L) >= this.maxOffsetLag;
            boolean skippedUpstreamRecord = upstreamOffset - this.previousUpstreamOffset != 1L;
            boolean bl = truncatedDownstreamTopic = downstreamOffset < this.previousDownstreamOffset;
            if (noPreviousSyncThisLifetime || translatedOffsetTooStale || skippedUpstreamRecord || truncatedDownstreamTopic) {
                this.lastSyncDownstreamOffset = downstreamOffset;
                this.shouldSyncOffsets = true;
            }
            this.previousUpstreamOffset = upstreamOffset;
            this.previousDownstreamOffset = downstreamOffset;
            return this.shouldSyncOffsets;
        }

        void reset() {
            this.shouldSyncOffsets = false;
        }
    }
}

