 /*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.crypto.examples;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Properties;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.crypto.cipher.CryptoCipher;
import org.apache.commons.crypto.cipher.CryptoCipherFactory;
import org.apache.commons.crypto.cipher.CryptoCipherFactory.CipherProvider;
import org.apache.commons.crypto.utils.AES;
import org.apache.commons.crypto.utils.Utils;

/**
 * Example showing use of the CryptoCipher API using a byte array
 */
public class CipherByteArrayExample {

    /**
     * Converts String to UTF8 bytes
     *
     * @param input the input string
     * @return UTF8 bytes
     */
    private static byte[] getUTF8Bytes(final String input) {
        return input.getBytes(StandardCharsets.UTF_8);
    }

    public static void main(final String[] args) throws Exception {

        final SecretKeySpec key = AES.newSecretKeySpec(getUTF8Bytes("1234567890123456"));
        final IvParameterSpec iv = new IvParameterSpec(getUTF8Bytes("1234567890123456"));

        final Properties properties = new Properties();
        properties.setProperty(CryptoCipherFactory.CLASSES_KEY, CipherProvider.OPENSSL.getClassName());
        // Creates a CryptoCipher instance with the transformation and properties.
        final String transform = AES.CBC_PKCS5_PADDING;
        byte[] output;
        int updateBytes;
        int finalBytes;
        Class<?> encipherClass;
        try (final CryptoCipher encipher = Utils.getCipherInstance(transform, properties)) {
            System.out.println("Cipher:  " + encipher.getClass().getCanonicalName());

            final String sampleInput = "hello world!";
            System.out.println("input:  " + sampleInput);

            final byte[] input = getUTF8Bytes(sampleInput);
            output = new byte[32];

            // Initializes the cipher with ENCRYPT_MODE, key and iv.
            encipher.init(Cipher.ENCRYPT_MODE, key, iv);
            // Continues a multiple-part encryption/decryption operation for byte array.
            updateBytes = encipher.update(input, 0, input.length, output, 0);
            System.out.println(updateBytes);
            // We must call doFinal at the end of encryption/decryption.
            finalBytes = encipher.doFinal(input, 0, 0, output, updateBytes);
            System.out.println(finalBytes);
            encipherClass = encipher.getClass();
            // Closes the cipher.
        }

        System.out.println(Arrays.toString(Arrays.copyOf(output, updateBytes + finalBytes)));

        // Now reverse the process using a different implementation with the same settings
        properties.setProperty(CryptoCipherFactory.CLASSES_KEY, CipherProvider.JCE.getClassName());
        try (final CryptoCipher decipher = Utils.getCipherInstance(transform, properties)) {
            System.out.println("Cipher:  " + encipherClass.getCanonicalName());

            decipher.init(Cipher.DECRYPT_MODE, key, iv);
            final byte[] decoded = new byte[32];
            decipher.doFinal(output, 0, updateBytes + finalBytes, decoded, 0);

            System.out.println("output: " + new String(decoded, StandardCharsets.UTF_8));
        }
    }

}
